package re2

import (
	"fmt"
	"math/rand"
	"runtime"
	"testing"
	"time"
)

// TestReplaceAllNoMatch is a regression test for https://github.com/wasilibs/go-re2/issues/56
func TestReplaceAllNoMatch(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}

	animalRegex := MustCompile(`cat`)

	var ms runtime.MemStats
	runtime.ReadMemStats(&ms)

	startAlloc := ms.HeapInuse

	for i := 0; i < 1000000; i++ {
		_ = animalRegex.ReplaceAllLiteralString(`The quick brown fox jumps over the lazy dog`, "animal")
	}

	runtime.GC()
	runtime.ReadMemStats(&ms)
	endAlloc := ms.HeapInuse

	if endAlloc > startAlloc && endAlloc-startAlloc > 1000000 {
		t.Errorf("memory usage increased by %d", endAlloc-startAlloc)
	}
}

// TestHighMem is a regression test for https://github.com/wasilibs/go-re2/issues/85
func TestHighMem(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}

	pattern := "\\b((103)\\d{16}|(9558|9111)\\d{15}|(95595|95596|95597|95598|95599|65)\\d{14}|(863529|862181|955100|861095|861798|861799|861797|860062|861285|862150|862151|862199|861096|862188|863218|861599|863698|863699|863686|861098|862180|862182|862187|862189|863676|863677|861674|862183|861622|861582|863219|860529|900000|861226|860516|861721|900010|861288|860086|861670|861225|861227|861281|862200|862202|862203|862208|861218|861476|863229|860058|861618|863260|863271|863272|861558|861559|861722|861723|861475|863062|861720|861379|860114|860146|861240|861724|861762|861414|861375|860187|862926|862927|862928|862929|862930|862931|861733|861732|861719|861369|861763|860046|863206|861671|860059|861282|861336|861619|862821|862822|862823|862824|862825|862826|862827|862828|862840|862841|862843|862845|862846|862848|862849|863018|863050|863053|863052|862844|862847|863056|862842|861667|861212|861283|860061|860040|861330|861331|861297|861620|861757|861672|863571|863572|863184|863586|863573|861665|867026|867028|867025|867027|861332|861333|861660|861661|861662|456351|601382|861256|861568|861569|861663|861666|861668|861669|861725|861756|861758|861759|861785|861786|861787|861788|861789|861790|863208|863263|863569|863575|862771|862772|862770|860531|862273|862274|860513|861284|436742|589970|860060|861081|861467|861621|861700|862280|862700|861598|861673|863211|863668|863094|863669|863656|863644|867020|860525|860521|862261|862260|862262|863560|863261|863191|860021|861069|861436|861002|861335|860527|862999|303|860535|861462|685800|860037|863506|863259|862568|860520|863078|861019|862309|863269|862933|862938|863031|863625|861016|861015|863187|861060|863551|861072|863630|861213|861290|861291|861292|863555|861289|861328|863176|861651|863163|863526|863565|863633|863622|863632|867061|867076|863019|861600|863514|863663|863561|863562|863283|863533|862865|863131|861418|860533|863252|862379|940008|861626|863058|862338|863205|863511|863112|863578|603445|862467|940016|861463|862455|940039|863118|863151|862993|863070|863069|863172|863173|603506|862499|940046|863000|862498|863582|861439|863010|863278|861751|861754|862143|940001|861532|861482|862135|862152|862153|862154|862996|862997|940027|862442|863099|863007|940066|862359|863531|862421|861417|863089|863200|861529|862429|862363|940048|861455|863119|863568|862877|862879|863291|861775|863203|863096|862425|861577|862133|862170|861735|861552|863098|861242|861496|863129|861538|863541|863275|861216|863179|861419|863170|860551|863212|69580|862162|863146|862307|863077|862857|940065|861237|863003|863039|940071|861446|862368|860519|861739|862967|940073|863127|862370|861437|863177|861415|861480|862147|861633|863057|862301|863171|861266|861422|863592|863631|862165|863102|862342|863048|862367|862392|863085|862395|862925|863595|862982|861413|861562|861097|861588|862644|863032|863518|863216|861037|862977|862897|863072|863285|861403|862561|862562|862563|863137|863264|862167|863193|863197|862532|861076|862131|862876|862873|861579|863279|862173|863509|862936|863060|862937|863101|863286|862980|863135|861726|861088|860517|862972|863196|863083|862973|863153|863121|861070|862978|863093|861200|863116|861038|863180|861086|861498|863290|861448|861296|861551|861044|863120|862940|861089|863161|861029|861766|863139|861071|863152|861074|861515|863030|861090|863178|861091|863168|861238|861057|863199|861233|863588|861223|861780|861239|863068|861273|863079|861325|863084|861366|861388|861348|860063|861360|863566|862959|863076|861391|861339|861469|861625|863113|863627|863293|863665|863688|861655|863537|861636|863182|863087|861696|863577|861460|861461|862855|863521|863138|861221|861748|862979|863037|863160|861035|861075|861271|861327|861337|861601|861753|863621|861688|863162|863552|862443|863132|940029|863071|603694|862323|861017|863190|940062|863038|862328|862184|861251|861589|863036|862127|861701|861560|861068|860088|861067|861066|862138|862531|862329|863103|863539|862343|861516|862345|863088|862452|861578|862324|863066|863267|862858|861058|861527|863091|861537|861736|863540|862869|863075|862882|861779|861590|862893|863090|861519|861539|862169|862681|862682|862683|863013|861021|862358|863025|863501|861566|862506|863027|863028|862992|863133|862935|861531|863181|863618|862947|861561|863095|861526|862953|861036|861458|861536|862509|861518|861728|862362|862470|862477|862510|861018|862859|861737|861533|861362|862976|861033|861099|861457|861459|861530|863201|863256|861061|861520|863065|861026|861449|862968|861280|861580|863051|861073|863109|863516|861228|861557|861361|863033|863207|861363|862891|863189|863576|863510|863603|863673|863628|863659|867070|863662|867060|861053|861230|861229|861481|861389|861411|861443|862409|861441|863107|862463|861087|861741|861782|863309|861743|861783|863308|861742|861784|863310|860072|862434|862436|862435|644|861253|861254|861255|861344|861274|861324|860126|860537|863459|863477|862546|862547|862548|862943|863318|921001|921002|860550|860559|921000|861563|860038|603367|863664|861752|861246|861727|863128|861778|861550|863220|863637|860528|860501)\\d{13}|(8601365|8601366|8601367|8601368|8601369|8630745|8630746|8630747|8630748|8630749|8635244|8635245|8635246|8635248|8651648|8651649|8658248|8658249|8630744|8635247|8601360|8635240|8630440|860200|860302|860402|860403|860404|860406|860407|860409|860410|860411|860412|860502|860503|860405|860408|860512|860602|860604|860607|860611|860612|860704|860706|860707|860708|860709|860710|860609|860712|860713|860714|860802|860711|860904|860905|860902|861103|861105|861106|861107|861102|861203|861204|861205|861206|861207|861208|861209|861210|861302|861303|861202|861305|861306|861307|861309|861311|861313|861211|861315|861304|861402|861404|861405|861406|861407|861408|861409|861410|861502|861317|861511|861602|861603|861604|861605|861608|861609|861610|861611|861612|861613|861614|861615|861616|861607|861606|861804|861807|861813|861814|861817|861901|861904|861905|861906|861907|861908|861909|861910|861911|861912|861913|861915|861903|862004|862005|862006|862007|862008|862010|862011|862012|861914|862015|862016|862018|862019|862102|862103|862104|862105|862013|862111|862114|862017|862110|862303|862304|862305|862306|862307|862308|862314|862315|862317|862302|862402|862403|862404|862313|862504|862505|862509|862513|862517|862502|862604|862605|862606|862510|862715|862806|862902|862706|863002|863006|863008|863011|863012|862904|863015|863100|863202|863301|863400|863602|863803|863901|863014|864100|864200|864301|864402|863700|864000|861001|861617|862002|862003|862020|862309|862703|862903|863500|553242|6858000|6858001|6858009|690755|966666|862909|862908|438588|438589|861298|863616|863661|863183|863185|861005|862172|862985|862987|862267|862278|862279|862468|862892|940021|861050|860522|861243|860041|861267|860043|863063|861664|990027|566666|603367|862878|863061|863209|603602|940053|861452|862331|863554|863574|940015|9896|940049|8880|8881|8882|8883|8884|8885|8886|8887|862981|863165|862136|863277|862485|861359|940017|694301|683970|940074|990871|862126|863166|862292|862291|861412|862960|863523|863653|861004|863130|863167|861345|861263|863080|861217|861270|861235|8624217|863667|9400301|909810|940035|861522|862439|863257|018572|940042|862991|861585|863013|863059|862271|940037|861392|861310|6506|6507|6508|6509|8634153|8634300|8634390)\\d{12}|(8601363|8630743|8635243|8651647|8658247|86926030|86926031|86926032|86926033|86926034|86926035|86926036|86926037|86926038|86926039|86926918|86926919|86926920|86926921|86926922|86926923|86926040|86926041|86926042|86926043|86926044|86926045|86926046|86926047|86926048|86926049|86926924|86926925|86926926|86926927|86926928|86926929|86926100|86926050|86926103|86926104|86926105|86926106|86926107|86926051|86926113|86926114|86926115|86926930|86926931|86926932|86926933|86926934|86926935|86926936|86926937|86926938|86926939|86926940|86926941|86215049|86215050|86215051|86218849|86218850|86218851|86326597|86329906|45806|53098|49102|53591|86329901|5453242|5491031|5544033|53242|53243|86364330|86364331|86364332|86329902|86153403|86364390|405512|601428|862259|86844808|862258|86364357|90030|86364873|95555|86326536|90592|863287|84380|84342|84385|84301|84336|84361|84373|84390|87000|87010|87030|87040|87050|862384|940034|86326528|86053803|86329900|86318799|86326603|86326594|86317622|86317633|86326516|86704300|86326584|86364305|86364306|86364307|86364312|86326612|86364308|86364310|86364309|86364316|86360066|86364323|86364339|86364353|6886592|86326580|86326575|86704309|86326531|940041|862420|862399|940043|603708|86326568|86326527|86326535|862442|86326555|86153402|940055|6091201|940056|96828|86326540|86326574|940057|862311|86326605|862990|940003|603601|862137|862327|862340|862366|862425|86326501|86326519|86364320|86326533|940006|862465|940031|86326538|86704307|86153405|86249802|94004602|862310|940068|863599|863670|940069|86326529|862400|86326582|862308|86326552|86326512|86326525|86329905|86704302|86249804|94004604|86326559|861760|862336|863273|86364352|86326571|86326599|861456|862856|863221|86326545|86326503|86326526|86326558|862962|86326515|86704305|86326613|86326587|86326602|86364303|86326564|86326617|862396|862511|86326610|86326530|86326523|862939|86326615|86326542|86326565|86364358|86364300|86326586|86326508|86326577|940044|86326583|86326591|86326561|862332|940063|863123|86326562|86326520|86326590|86326573|86153401|86326608|86326566|86153406|86326550|86326537|86326539|86704304|86153404|86326601|86326502|86326578|86364318|86364328|86105600|86105601|86359701|86355603|86355602|86355601|86359704|86359703|86359702|86109209|86109207|86109202|86109203|86109208|86109211|86109205|86109210|86109212|86109206|86109214|86109215|86109213|861308|861219|86134615|86134614|86134617|86134616|86134605|86134608|86134620|86134603|86134604|86134609|86134622|86134621|86134602|86134601|86134613|86134625|86134611|86134624|86134606|86134626|86134607|86134618|86134619|86134627|86134623|86325409|86325405|86325407|86325403|86325404|86325410|86325408|86325406|86707549|86707533|86707534|86140101|86163751|86163743|86163723|86163728|86163727|86163724|86163725|86163750|86308299|86319801|86319806|86319802|86319803|86309702|86309701|86165666|86165932|86169301|86169302|86169303|86169502|86169505|86169501|86169507|86169503|86169504|86169506|86352801|86311702|86312201|86317502|86317501|86138501|86316904|86316905|86316902|86316903|86316901|86316906|86138402|86138404|86138403|86138401|86138407|86138405|86138408|86138406|86138409|86138410|86360636|86360637|86360634|86360630|86360632|86360631|86360633|86360635|86360638|86360602|86360603|86360629|86360625|86360626|86360628|86360627|86360601|86360620|86360622|86360624|86360623|86360621|86360600|86360607|86360608|86360605|86360604|86360606|86360609|86360612|86360613|86360610|86360614|86360615|86360616|86360611|86360617|86360618|86360619|86350801|86109216|86351501|86168305|86168301|86168304|86168302|86168309|86168308|86168306|86168303|86168307|86352501|86352700|86352702|86352701|86314911|86314907|86314903|86314906|86314909|86314902|86314901|86314910|86314908|86314905|86357012|86357017|86357009|86357022|86357021|86357015|86357013|86357010|86357014|86357006|86357002|86357008|86357011|86357007|86357016|86357004|86357003|86357018|86357001|86357005|86357019|86357020|86314701|86358101|86358303|86358301|86358302|86361702|86361703|86361701|86143801|86143802|86006401|86300402|86300401|86310402|86310401|86364201|86363500|86361902|86361901|86362401|86365003|86365002|86365001|86365005|86365006|86365007|86365008|86365011|86365010|86365009|86359602|86359601|86359606|86359607|86359603|86359605|86707305|86707301|86707302|86707306|86707308|86707303|86707307|86707304|86165203|86165202|86165201|86165204|86304701|86365401|86365227|86365225|86365229|86365226|86365202|86365228|86365220|86365213|86365224|86365217|86365221|86365215|86365218|86365223|86365222|86365214|86365216|86365211|86365219|86365210|86365212|86365203|86365204|86365205|86365208|86365209|86365206|86365207|86365504|86365502|86365501|86365503|86365506|86365507|86365701|86311401|86311402|86354903|86354902|86367201|86134612|86367401|86364001|86364602|86364607|86364603|86364608|86364605|86364604|86364601|86364606|86364921|86368401|86368701|86363801|86367502|86367503|86367501|86706401|86706501|86367901|86147000|86707501|86353400|86315400|86315402|86315401|86315403|86315404|86177400|86177401|86177403|86177404|86177402|86354702|86354701|86354700|86313401|86313403|86313400|86313402|86707506|86316401|86316400|86704114|86704101|86704000|86704001|86704003|86704002|86707503|86707508|86707507|86704201|86704203|86704205|86704202|86704200|86704204|86707505|86707504|86707517|86707519|86707510|86707542|86707535|86707539|86707536|86707512|86707540|86707541|86707537|86707538|86707516|86707518|86707521|862365|862375|862489|862487|862490|862491|862492|861744|861745|861746|861747|862971|862970|863497|861442|862410|861440|863106|862459|863328|863397|863465|86911022|60110|60112|60113|60114|60119|86911054|86911055|86341550|86340800|86921000|86921001|86925200|86911029|86917901|86921002|86293423|86913610|86911200|86911052|86911053|86909504|86911700|86911506|86911063|86909503|86053801|86053802|86053804|86326551|86911015|86364391|86326616|86364993|86707523|86707777)\\d{11}|(865153|8601362|8630742|8635242|8651646|8658246|86926020|86926021|86926022|86926023|86926024|86926025|86926026|86926027|86926028|86926029|86926912|86926913|86926914|86926915|86926916|86926917|427062|427064|862246|544210|548943|402791|427028|427038|548259|524374|550213|862597|862599|865865|865866|865899|510529|524091|865650|862910|862911|862912|862913|865160|865161|865162|865651|356879|356880|356881|356882|427010|427018|427019|427020|427029|427030|427039|438125|438126|451804|451810|451811|458071|458441|489734|489735|489736|513685|524047|525498|526836|528856|530970|530990|543098|558360|862206|862210|862211|862212|862213|862214|862215|862220|862223|862224|862225|862229|862230|862231|862232|862233|862234|862235|862237|862238|862239|862240|862245|865330|865331|865332|865708|865709|868286|868288|862299|865247|865249|865250|865858|865859|865860|865248|862236|865929|861376|860054|860142|861423|865927|861428|865939|861434|865987|861761|861749|860184|865930|863321|861300|861378|865114|862159|865021|865022|865932|862889|865900|862949|865915|865916|862944|865115|865941|865801|862171|865931|865918|865113|861371|860143|860149|861730|865928|865914|861734|861433|865986|861370|865925|860124|860183|860561|865116|862227|865017|865018|865019|865921|861764|863034|865110|861464|865942|862158|865917|860094|860186|861719|861765|865922|865934|861750|865933|861377|860148|860185|865920|861367|865924|861731|861781|552599|403361|404117|404118|404119|404120|404121|463758|514027|519412|519413|520082|520083|558730|862836|862837|865996|865997|865998|868268|862839|865826|548478|544243|862820|862830|862838|865336|868269|865653|868346|865171|865170|865827|865818|865999|356833|356835|409665|409666|409668|409669|409670|409671|409672|512315|512316|512411|512412|514957|409667|518378|518379|518474|518475|518476|438088|524865|525745|525746|547766|552742|553131|558868|514958|862752|862753|862755|524864|862757|862758|862759|862760|862761|862762|862763|862756|868388|860514|862754|518377|862788|558869|865905|865906|865907|868313|865333|868312|865337|865338|865568|865908|865909|865910|862764|862765|865834|866201|861648|869143|860210|860211|862479|862380|866200|862480|860019|860035|862789|861638|861334|865140|861395|862725|862728|421349|434061|434062|436728|453242|491031|524094|526410|544033|552245|861080|861466|861488|861499|862707|862966|862988|865955|865956|861082|863251|436742193|862280193|356896|356899|356895|436718|436738|436745|436748|489592|531693|532450|532458|544887|552801|557080|558895|559051|862166|862168|862708|865964|865965|865966|868266|868366|865362|865363|868316|868317|434910|458123|458124|520169|522964|862250|862251|521899|862254|862255|862257|862253|862656|868216|862252|955590|955591|955592|955593|868218|491040|537839|865138|862256|862284|862285|860013|865028|865029|433670|433680|442729|442730|860082|862690|862691|862692|862696|862698|862998|433671|968807|968808|968809|861771|861767|861768|861770|861772|861773|861769|863280|862662|862663|862664|862665|862666|862667|862669|862670|862671|862672|862668|862661|862674|862673|860518|861489|861492|863156|861490|861491|860085|863155|863157|863158|863159|863253|862660|999999|863020|863021|863022|863023|862630|862631|862632|862633|863292|860552|861222|862615|862616|862618|862622|862617|862619|415599|421393|421865|427570|427571|472067|472068|862620|861691|900003|863683|861692|863255|863258|545392|545393|545431|545447|356859|356857|407405|421869|421870|421871|512466|356856|528948|552288|862600|862601|862602|517636|862621|868258|556610|862603|464580|464581|523952|553161|356858|862623|865911|865912|865913|865188|406365|406366|428911|436768|436769|487013|491032|491034|491035|491036|491037|491038|518364|520152|520382|548844|552794|862555|862556|862557|862558|862559|528931|558894|865072|865071|865805|865806|865807|865808|865809|865810|862560|868259|868260|862890|865073|863365|863366|412963|415752|415753|862538|998800|412962|860010|862535|862536|862539|862983|356885|356886|356887|356888|356890|439188|439227|479229|521302|356889|552534|862575|862576|862581|868290|862578|402658|410062|468203|512425|524011|545619|545620|545621|545623|545948|552587|861286|861483|861485|861486|862577|862579|862580|862588|862609|863126|863136|865657|863262|863563|479228|545947|861276|862582|862598|862585|439225|518710|518718|868362|439226|868262|865802|865803|861299|461982|486493|486494|486861|523036|451289|527414|528057|862901|862902|862922|868212|451290|524070|865084|865085|865086|865087|548738|549633|552398|865082|865083|865960|865961|865962|865963|865353|865356|862571|862572|862573|862591|862592|862593|356851|356852|404738|404739|456418|498451|515672|356850|517650|525998|862177|862277|862516|862517|862518|862520|862521|862522|862523|868222|984301|984303|862176|862276|862228|861351|861390|861792|865957|865958|862519|861793|861795|861796|862500|863250|861352|863559|861791|863658|860530|865821|868324|861268|862884|861453|862684|865122|861062|861063|865076|865077|865074|865075|862946|865102|862942|862994|862583|862584|862950|862951|861201|863513|861077|861245|861277|86707866|86707877|86707888|86707899|356827|356828|356830|402673|402674|438600|486466|519498|520131|524031|548838|862148|862149|862268|356829|862300|868230|862269|865099|865953|865350|865351|865352|519961|865839|865180|862393|940023|421317|602969|861420|861468|863111|422160|422161|861030|862886|862388|868360|862695|940012|862178|868358|862179|862394|940025|861279|862281|862316|940022|602907|862986|862989|862298|940032|861977|862325|863029|863105|862475|861244|863081|863108|862955|865811|868285|862466|868309|862383|862385|868299|86326513|865800|868242|862595|862303|862596|862305|861259|862333|940050|940051|868204|862449|863067|862450|868278|865502|865503|865135|862476|865155|862486|863026|863086|868291|868302|862397|862398|940054|863598|868205|862426|865995|868214|865529|865335|862428|861776|863016|862134|940018|60326513|940060|862415|868329|865190|940040|862139|861269|862275|940061|861252|862146|862440|940047|862418|940002|862413|863188|868357|862321|865001|868273|862427|940070|865901|862898|868281|868282|862806|868283|862900|862391|940072|868319|862132|861340|861341|863073|862140|862141|868395|862335|863001|862315|865359|865950|868295|862165|862337|862411|862448|862860|862870|863150|862866|862880|862881|860118|868279|862864|862777|861497|862888|868398|862868|862899|868255|865988|862626|865946|868200|862566|862567|862625|504923|862422|862447|940076|531659|862157|435744|435745|483536|862525|862526|998801|998802|528020|862155|862156|526855|356869|356868|865360|865361|868296|865825|865823|868210|862283|865902|861010|868351|862740|865036|861014|861028|868250|868253|868378|868385|861755|862945|868355|868339|868325|868303|868389|868380|868315|861738|861272|868331|865850|868340|868356|865201|861591|862961|862508|863660|863505|862441|863623|868377|867069|867066|867067|867068|862478|861495|940013|865101|862462|868272|862128|863035|863186|863296|862129|940038|985262|862322|86326581|862369|862412|861523|863055|861013|865889|868254|863276|862312|868381|862481|862341|940058|863115|940020|861465|861528|861258|862867|862885|900105|900205|861521|861690|862320|868375|862319|865288|865888|868298|865186|868336|865526|861024|862339|860500|862289|862389|868300|865516|868248|862648|862488|863110|862686|862288|868280|868247|865182|862895|863125|868323|865121|868330|861008|861525|861287|868277|865150|868326|865225|865226|868387|861065|863017|863213|861517|868289|862302|868297|865097|868332|865183|868399|861449082|865126|868320|861018139|861018140|861018141|861518091|861518092|861518093|861728091|861728092|861728093|86364350|861056802|861056805|861056801|861056803|862995|862975682|862975681|861260119|861260103|861260121|861260123|861260105|861260117|861260113|861260107|861260108|861260109|861260111|861260112|861260115|861260106|861260110|861260116|861260122|861260120|861260002|861260102|861260001|861260118|861260136|861260125|861260132|861260127|861260130|861260135|861260133|861260128|861260131|861260129|861260126|861275341|861275171|861275181|861275291|861275281|861275111|861275191|861275121|861275321|861275221|861275201|861275251|861275161|861275131|861275241|861275271|861275231|861275141|861275261|861275101|861275151|861275361|861275351|861275301|861275371|861275211|861275311|861275331|861092007|861092003|861092002|861092001|861092008|861092006|861092004|861092005|861092011|861092012|861092010|861241001|861241003|861241002|861241004|862218|868267|861346003|86354320|861346002|861346102|861346001|86354332|86354331|861326919|861326969|861326979|861326878|861326989|861326939|861326888|861326626|861326898|861326999|861326828|861326959|861326858|861326848|861326868|861326929|861326838|861326818|861326949|861326762|861338005|861338008|861338001|861338007|861338003|861338002|861338010|861338006|861338009|861338011|861353121|861353008|861353108|861353113|861353115|861353117|861353002|861353102|861353005|861353105|861353007|861353107|861353003|861353103|861353006|861353106|861353109|861353001|861353101|861353124|861353110|861353112|861353111|861353123|861353122|861353118|861353119|861353120|861356092|861356094|861356888|861356124|861356095|861356083|861356125|861356123|861356118|861356032|861356120|861356034|861356121|861356116|861356119|861356040|861356117|861356111|861356014|861356038|861356013|861356112|861356106|861356108|861356109|861356110|861356107|861356091|861356113|861356055|861356046|861356044|861356030|861356026|861356016|861356015|861356029|861356047|861356059|861356054|861356027|861356101|861356105|861356023|861356084|861356005|861356102|861356103|861356018|861356048|861356024|861356104|861356006|861356004|861356003|861356017|861356036|861356049|861356025|861356051|861356020|861356019|861356070|861356071|861356021|861356114|861356086|861356087|861356050|861356007|861356009|861356008|861356002|861356045|861356062|861356001|861356061|861356010|861356012|861356042|861356011|861356085|861356122|861356096|861356089|861356033|861356035|861356053|861356056|861356063|861356088|861356028|861356037|861356052|861356069|861356074|861356058|861356072|861356041|861356039|861356073|861356022|861356067|861356079|861356080|861356078|861356081|861356068|861356082|861356066|861356065|861356077|861356057|861356060|861356031|861356100|861356097|861356099|861356098|861356043|861356076|861356075|861356115|861356064|861356090|861356093|861347102|861347108|861347105|861347103|861347101|861347106|861347107|861350010|863589010|863589020|861350020|861350431|863589431|863589001|863589451|861350451|861350001|863589027|861350013|863589013|863589026|863589005|861350005|863589021|861350009|863589009|861350003|863589003|861350002|863589002|861350015|861350004|863589004|863589015|861350006|863589006|861350011|863589011|861350016|863589016|861350007|863589007|861350755|863589755|863589017|861350017|863589025|863589024|861350014|863589014|863589028|861350019|863589012|861350012|861350008|863589008|861350018|863589019|863589018|861350943|863589943|863589023|863589022|861399011|861365006|861365001|861365012|861365011|861365005|861365013|861365014|861365007|861365004|861365009|861365008|861365003|861365002|861365010|861401001|861401002|863143005|863143003|863143001|863143004|863143002|861393001|86139383|861623001|861397102|861397001|861397101|861627008|861627005|861627002|861627001|861627007|861627009|861627003|861627006|861627010|861635101|861635114|861635003|861635103|861635124|861635004|861635104|861635112|861635109|861635111|861635013|861635113|861635119|861635010|861635110|861635005|861635105|861635007|861635107|861635118|861635122|861635115|861635108|861635117|861635125|861635123|861635120|861635106|861635102|861635116|861650105|861650104|861650003|861650002|861650102|861650001|861650101|861630002|861630102|861630003|861630103|861630001|861630101|86163113|86163103|86163122|86325400|86163119|86163120|86163112|86163117|86163114|86163116|86163115|86163104|86163118|86163108|86325401|86163127|86163107|86163121|86163125|86163101|86163123|86163126|86163102|86163109|86325402|86163110|86163111|861637009|861637008|861637006|861637007|861637005|861637004|861637001|861637002|861637003|861653002|861653003|861653004|861653005|861653007|861653006|861653001|861628661|861628662|861628660|861675002|861675001|861675004|861675005|861675003|861656002|861396|861656001|861659002|861659001|86165934|861659006|861659009|86165954|861659008|861659010|861659003|861398001|861398002|861676021|861676014|861676009|861676010|861676001|861676002|861676003|861676006|861676024|861676035|861676025|861676013|861676012|861676023|861676011|861676008|861676040|861676037|861676038|861676039|861676036|861676041|861676005|861676026|861676004|861676027|861676029|861676028|861676015|861676019|861676020|861676030|861676022|861676031|861676032|861676046|861676018|861676033|861676045|861676044|861676042|861676043|861676048|861676047|861676034|861676016|861676017|861678108|861678102|861678105|861678103|861678101|861678601|861678106|861678107|861678001|861678501|861678110|861678002|861678502|861678109|861678128|861678127|861678126|861678125|861678301|861678208|861678708|861678213|861678713|861678207|861678707|861678209|861678709|861678211|861678711|861678113|861678212|861678712|861678112|861678111|861678202|861678702|861678201|861678701|861678203|861678703|861678205|861678705|861678206|861678706|861678117|861678116|861678210|861678710|861678119|861678121|861678118|861678124|861678215|861678715|861678122|861678115|861678120|861678114|861678123|861679004|861679013|861679012|861679008|861679005|861679001|861679003|861679002|861679007|861679006|861679009|861679010|861679011|861681101|861681005|861681102|861681103|861682002|861682801|861682839|861682101|861682803|861682102|861682805|861682106|861682808|861682103|861682806|861682105|861682807|861682110|861682812|861682111|861682813|861682109|861682811|861682108|861682810|861682107|861682809|861682202|861682816|861682201|861682815|861682203|861682817|861682205|861682818|861682209|861682822|861682208|861682821|861682210|861682823|861682213|861682827|861682211|861682825|861682212|861682826|861682207|861682820|861682206|861682819|861682003|861682802|861682301|861682828|861682302|861682829|861682305|861682831|861682307|861682833|861682306|861682832|861682309|861682836|861682308|861682835|861682310|861682837|861682303|861682830|861682311|861682838|861685701|861685702|861687001|86169757|86169724|86169734|86169769|86169768|86169767|86169716|86311703|86311701|861380001|861381001|861382019|861382018|861382020|861382001|861382002|861382006|861382005|861382010|861382009|861382007|861382008|861382003|861382004|861382025|861382013|861382017|861382021|861382023|861382015|861382016|861382012|861382014|861382024|861382027|861382011|861382022|861382026|86319563|861383001|861383002|861265001|86127852|86127834|86127843|86127825|86127839|86127861|86127858|86127827|86127823|86127877|86127822|86127821|86127828|86127820|86127836|86127837|86127831|86127832|86127855|86127810|86127816|86127817|86127812|86127815|86127813|86127801|86127800|86127818|861384001|861368001|861386002|861386001|861386003|861386004|861699008|861699002|861699007|861699001|861699005|861699003|861699006|863601001|863604001|863608001|863608002|863609001|863609002|863609003|863609006|863609005|86361026|86361027|86361025|86361028|863512001|863045001|863607012|863607006|863607014|863607015|863607011|863607008|863607013|863607007|863607002|863607003|863607004|863607010|863607001|863607005|863607009|863522001|863504010|863504002|863504008|863504004|863504009|863504001|863504005|863517001|863504003|863504006|863504007|863525003|863525004|863525022|863525005|863525006|863525025|863525023|863525024|863525009|863525021|863525020|863525002|863525008|863525007|863525001|86352739|86352736|86352760|86352755|86352733|86352759|86352737|86352784|863532001|863145001|86314061|86314062|863538001|86355001|86355358|863558617|863558618|86355865|863558621|863558615|863558622|863558611|863558616|863558612|863558613|863558620|863558619|86358421|86359430|86368004|86368001|86368002|86368003|863580001|863579001|86358563|863587001|863587002|86359065|86359364|86359354|86359356|86359350|86359353|86359329|86359349|86359351|86144735|861658001|86165826|86361345|86361346|86361347|86314292|86361528|86361527|86361560|86361523|86361522|86361525|86361526|863626706|863626704|861382076|861382070|861382053|861382077|861382066|861382078|861382063|861382052|861382068|861382074|861382051|861382057|861382058|861382080|861382061|861382062|861382067|861382060|861382072|861382064|861382056|861382050|861382075|861382073|861382079|861382069|861382071|861382054|861382065|861382059|861382055|86363585|86363440|86363441|86363442|86361912|86361982|86317475|86354290|86354211|86354215|86354213|86165271|86365178|86354901|863567001|863567002|86363914|86363938|86363974|86363975|86363940|86363971|86367292|86367291|86364710|86314289|86314259|86364178|86364194|86354884|86354897|86354879|86369470|86707403|86707411|86707405|86707406|86707407|86707410|86707409|86707408|86707401|86707402|867063003|867063001|867063002|86367933|86367934|86367935|86367936|86360530|86149445|86316424|86316420|86316438|86316448|86316446|86316443|86316483|86316442|86704119|86704118|86707522|86707530|86707529|86707527|86707531|86707528|862372|862471|862472|862371|865091|862293|862295|862296|862297|862373|862451|862294|865940|862871|862958|862963|862957|862798|865010|862381|862382|861487|861083|862487|862490|862491|862492|861034|862386|862952|865107|862387|862423|865062|865063|862360|862361|865034|865096|865098|864326|865026|865024|862376|862378|862377|865092|862453|862456|864303|865008|865009|864337|866396|865055|865040|865042|865044|865046|865048|865053|865058|865060|865136|865172|865174|865176|864472|864483|866206|866208|866210|866212|866214|865141|865143|862750|862751|865145|865147|865196|865198|864474|864485|863040|860202|860203|862348|869225|863041|860208|860209|862351|863042|862355|862493|864489|860204|860205|861084|863350|864412|862948|861740|862482|862483|862484|863334|869131|869140|864468|866407|864477|864478|864481|869139|864488|869148|869238|869239|864458|862675|862676|862677|864329|869173|869174|869175|866240|869221|920232|920239|920332|920339|920432|920439|941643|941725|944022|944631|944664|945903|865842|862433|862861|862932|862862|862775|862785|864300|864302|864308|864309|862920|864407|864408|863043|863064|864371|864372|864398|862403|601100|601101|601121|601123|601124|601125|601126|601127|601128|601122|601131|601136|601137|601138|601140|601142|601143|601144|601145|601146|601147|601148|601149|601174|601178|601186|601187|601188|601189|601177|601179|865014|865016|862549|862550|863357|864341|862354|863488|861224|862954|861295|865124|865154|863457|862444|862414|866221|861049|862445|861000|860011|860027|860031|860039|860103|860106|860123|860125|860220|860278|860812|861006|861025|861027|860132|865003|865011|865012|865020|865025|865027|865031|865039|865078|865079|865120|865127|865032|865244|861484|861640|861641|865245|865246|864344|863395|864384|864345|863391|863392|864381|864382|863451|863452|863453|864448|864375|864482|869141|869149|869150|861639|863394|866415|866416|864451|864452|865006|865023|865103|865106|865111|865112|865123|865125|865131|865132|865139|865178|865243|861079|861003|861011|861012|861023|861031|861032|861039|861078|865242|866275|865241|864449|864450|864414|864347|864403|864406|864383|866422|861045|862356|861234|862145|865013|862130|861257|861055|860009|865002|865033|865035|865007|860015|860024|865004|861349|860108|861354|863326|863327|860532|864313|864333|863358|865904|864314|866428|861645|866233|866234|866235|869176|869177|861624|863332|864338|864351|863339|863370|864435|864436|866399|866400|866409|866236|866237|866261|861642|864366|861654|865804|863374|864370|864434|866402|863489|864358|864462|860056|865814|865817|864399|864400|864401|864410|864411|866395|864413|864440|864441|864442|864443|866222|864444|863316|863317|863315|860079|860091|864357|861657|863024|865840|865841|864331|864348|864332|866394|864339|864432|864479|864480|861794|865944|863347|864428|864429|866398|864306|864322|863300|863302|863303|863304|863324|863492|863398|863307|863311|863312|863313|863323|863341|864320|864321|864324|864325|864387|864388|864349|864350|864335|864359|864360|864362|864363|866344|863314|864490|863325|863375|869144|863331|863348|863336|863337|863338|864323|863360|863344|863499|863493|863359|863362|863364|864352|864353|864354|863381|863402|863380|864367|863385|863383|863386|863388|864368|864376|864377|864378|863389|863417|863418|863419|863420|863421|869180|863410|863411|864392|864393|864394|864395|864396|861052|863414|864391|863448|863443|863355|866289|866293|866294|864416|864418|860120|861020|861220|863379|863393|863454|863470|864343|864346|864373|864374|864380|864404|864420|864424|864437|864446|865220|865320|866250|866251|866277|864336|864361|866426|864389|864427|864471|864334|864423|864425|864459|863456|864455|869142|866202|869152|866219|866412|866413|869166|869167|869283|866231|869172|866297|862194|869215|869217|860129|866279|865945|862346|862347|862349|862350|862352|862353|860515|860048|920000|862812|862810|862811|868310|865919|865368|865367|518905|862835|865603|865605|865206|865052|865215|400360|403391|403392|404158|404159|404171|404172|404173|404174|404157|433667|433669|514906|403393|520108|433666|558916|862680|862688|862689|868206|556617|868209|518212|868208|356390|356391|356392|862916|862918|862919|868370|868371|868372|862766|862767|862768|862678|862679|862657|862685|862659|862687|865981|865979|356839|356840|406252|406254|425862|481699|524090|543159|862161|862570|862650|862655|862658|865975|868201|868202|865976|865339|862801|865978|865980|865974|862800|865977|523959|528709|539867|539868|862637|862638|868318|528708|862636|865967|865968|865969|865971|865970|865831|862693|865833|865993|868221|865191|862265|862266|865973|862883|865972|865093|865095|522001|862163|862853|868203|862851|862852|865816|865165|868314|865169|865903|862778|868207|862282|862318|512431|520194|868379|865890|865050|865836|868367|862875|868333|865208|868349|862921|868321|865598|862286|868236|861777|863270|862813|862818|868228|865166|868359|868274|865163|868341|865216|868270|868311|862710|868294|868261|868251|862651|865828|865652|865700|865358|862613|868220|862809|868224|865119|865577|865952|868213|862533|862537|865167|868345|862416|868263|865370|862718|868305|868361|865202|868239|865195|868238|868257|868304|862817|868287|865959|865129|868391|868233|868264|868231|865157|868353|865187|865209|868327|865205|865203|868275|862565|862717|862287|868252|868306|868227|868234|868337|868237|868365|868219|865168|865369|862270|865090|868368|865588|865212|868373|868293|865192|868276|862611|868211|865500|865989|865701|862722|865838|868243|865080|868235|868322|865088|862469|868307|868229|868397|862802|862290|868232|865128|865357|862829|865819|868301|862808|868308|862815|862816|868226|865189|868223|861416|868217|868284|865820|868382|865158|862569|868369|865506|862906|868386|865519|863092|868392|865992|868244|868364|868350|865133|865152|868342|865156|868271|865886|868328|865891|868347|865200|868343|865837|868354|861293|861294|861342|861343|861364|861394|863423|869153|863485|861248|869163|861215|861249|865829|865943|861231|862790|864405|862164)\\d{10}|(8601361|8630740|8630741|8635241|8651640|8651642|8651643|8651644|8651645|8658240|8658241|8658242|8658243|8658244|8658245|8651641|8601364|8635249|86926010|86926011|86926012|86926013|86926014|86926015|86926016|86926017|86926018|86926019|86926906|86926907|86926908|86926909|86926910|86926911|8628119|8645806|8653098|374738|374739|370246|370247|370248|370249|370267|8643190|8643191|8643192|8643193|8644220|8644221|8644222|8644223|6349102|6353591|377677|8633670|8633671|8633672|8633673|8633674|8633675|8634340|8634341|8634342|8634343|8634344|8634345|00405512|00601428|66405512|66601428|377152|377153|377158|377155|690755|370285|370286|370287|370289|8623091100|8623092900|8623093310|8623093320|8623093330|8623093370|8623093380|8623096510|8623097910|86326576|8629756114|8629756115|861326763|861392887|861681001|861681002|861681003|861687913|861697813|861697793|861697873|861689007|861689001|861689002|861689004|861689005|861689006|861689003|861387973|861381002|861381003|861385663|861278193|861278293|861278503|861278333|863678353|863502001|863519533|863530963|863530953|863194773|861652853|8658433|8658434|6011290|6011291|6011292|6011293|6011295|6011297|6011294|6011296|6011300|6011390|6011392|6011393|6011391|6011399|8634500|8634501|8664030|8664031|8664032|8644950|8644951|8644952|8644953|8644954|8633710|8633712|8633720|8633722|8633723|8633730|8633732|8633760|8633762|8643650|8643652|8643651|8633711|8633721|8633731|8633761|8634280|8634290|8650386|8601086|8601088|8616846|8616848|8650388|8634610|8692740|8644760|8633451|8633452|8633453|8633454|8633455|8614455|8614458|8634240|8634241|8634242|8644570|8633493|8633494|8643051|8643052|8662740|8643550|8643551|8643560|8643561|8662440|8662450|8662451|8633603|8633604|8633605|8633461|8633462|8633463|8621441|8643420|8643421|8643422|8643423|8643424|8643425|8633681|8621740|8633840|8633841|8633820|8633821|8633822|8633823|8633824|8633960|8633961|8634460|8634461|8634462|8634470|8634211|8691802|8634090|8643900|8643901|8643902|8633990|8633991|8634150|8634151|8634152|8634154|8634250|8634251|8634252|8634253|8634220|8634270|8634320|8634321|8634322|8634323|8634350|8634351|8634352|8634420|8634421|8634422|8634450|8634451|8634452|8634453|8644160|8634440|8634441|8692710|8692711|8634670|8663740|8634671|8663741|8692720|8692790|8692791|8634750|8634620|8634621|8634622|8634623|8644650|8644651|8644652|8644653|8662580|8662581|8644700|8644701|8644702|8664080|8664081|8664082|86349600|86349100|86349101|8662300|8662180|8662181|8662182|8662183|8662184|8662185|8662490|8662480|8662550|8662530|8662560|8662570|8662730|8662720|8692360|8662820|8662830|8662910|8662920|8634330|376968|376969|376966|377187|8627890|8627891|8627892|8627893|8627894|8627895|8627896|8627897|8627899|8627898)\\d{9}|(86636000|86636001|86636002|86636003|86636004|86636005|86636006|86636007|86636008|86636009|86926000|86926001|86926002|86926003|86926004|86926005|86926006|86926007|86926008|86926009|86926900|86926901|86926902|86926903|86926904|86926905|86636010|86636011|86636012|86636013|86636014|86636015|86636016|86636017|86636018|86636019|86636100|86636020|86636021|86636101|86636102|86636103|86636104|86636105|86636106|86636111|86636200|86636119|86636120|86636121|86636122|86636123|86636124|86926111|86926112|86926201|86636022|86636023|86636201|86636202|86636203|86636204|86636205|86636206|86636207|86636208|86636209|86636210|86636211|86636212|86636125|86636126|86636127|86636128|86926116|86926117|86926118|86926119|86636129|86636130|86636131|86636132|86926202|86636133|86636134|86636135|86636136|86636137|86636138|86926052|86926053|86451804|86451810|86451811|86458071|360883|360884|86844801|86137310|86137320|86592310|86592320|86592340|86013101|86013102|86176410|86176420|86592110|86592120|86592140|86594250|86594260|86594270|86629621|86431940|86442240|86844803|86844802|86589700|86560602|86589710|86589711|86844800|86911099|86519391|86519392|86928600|86928601|86152400|86519326|86599400|86565801|86326510|86844811|86519333|86519315|86599506|86511900|60326500|86326522|86560600|86565800|86228306|86228308|86844805|86519331|86364302|86519311|86519317|86519327|86844706|86844810|86326611|86231902|90010502|90020502|86519337|86519329|86844707|86519313|86844709|86364324|86519325|86844705|86519336|86844710|86105913|86105903|86105916|86105906|86105915|86105905|86105901|86105911|86105900|86105902|86519306|86826788|86515901|86355701|86355702|86355703|86312405|86312401|86312406|86312402|86312403|86911050|86431001|86626000|86909502|86911051|86911059|86911060|86293425|86629502|86629500|86629501|86629503|86629504|86629505|86628601|86925100|86925101|86629632|86628000|86628001|86629600|86629633|60112010|60112011|60112012|60112089|60112013|60112980|60112981|60112986|60112987|60112988|60112989|60112990|60112991|60112992|60112993|60112996|60112997|60113080|60113081|60113089|60112995|60113940|60113941|60113943|60113944|60113945|60113946|60113984|60113985|60113986|60113988|60112994|60113989|86629602|86449550|86345800|86927700|86345801|86345802|86345803|86927701|86445400|86293410|86624910|86624911|86922400|86349550|86235600|86915500|86915501|86915502|86915503|86925600|86915504|86629800|86629801|86913200|86446600|86624100|86624101|86910100|86910101|86910102|86910103|86910104|86910120|86910121|86447610|86447611|86447612|86447613|86445300|86445301|86924200|86621701|86641002|86641001|86443100|86443101|86911028|86927300|86921600|86334910|86334911|86334912|86334921|86334923|86334913|86334950|86334951|86629695|86916501|86916500|86641000|86621700|86335201|86335202|86335203|86911071|86911505|86335101|86335102|86335103|86335104|86335105|86335106|86335107|86335108|86441900|86441901|86441902|86441903|86629608|86629609|86911014|86348700|86928400|86293430|86348702|86348701|86340500|86622700|86622701|86622702|86629660|86629661|86629662|86642515|86916100|86916101|86622800|86929505|86341403|86341407|86341408|86341409|86341401|86341402|86629679|86344901|86441701|86347800|86927800|86344902|86927801|86915700|86438500|86343800|86344000|86344001|86441500|86441501|86344100|86345500|86346000|86346001|86340410|86340420|86348100|86348101|86346400|86442600|86442601|86346600|86443000|86443001|86443002|86446300|86640400|86924400|86629607|86924401|86924402|86347600|86444500|86444501|86444502|86444503|86913000|86913002|86913003|86622300|86913004|86348000|86348001|86629000|86340610|86340620|86340630|86929512|86446700|86446720|86446710|86347900|86914700|86928100|86446900|86921400|86913800|86913300|86622000|86620401|86620402|86915800|86913500|86624301|86921200|86915100|86917800|86915101|86915400|86915401|86915402|86915403|86449600|86449601|86449700|86449701|86641100|86621600|86621601|86912901|86623900|86912800|86912900|86330500|86620500|86916200|86916201|86438600|86438601|86916400|86916401|86916402|86917100|86622500|86622600|86224100|86916800|86629604|86629605|86642500|86911008|86929500|86916900|86340700|86340701|86340705|86340706|86340100|86622900|86623200|86641400|86623201|86623202|86641401|86911025|86911026|86911027|86623800|86624200|86624201|86921100|86917900|86629610|86629611|86629612|86629613|86624700|86624701|86624702|86641800|86641801|86641802|86641803|86641804|86348300|86348301|86929509|86330502|86330503|86348400|86625900|86625901|86625400|86922000|86922200|86922201|86922001|86922700|86629615|86629603|86629673|86922300|86925500|86925401|86922800|86922801|86923000|86923001|86928900|86928901|86924900|86627600|86642300|86922600|86928700|86629645|86629646|86629647|86911041|86911042|86911043|86911044|86911045|86911046|86629623|86911056|86911057|86911058|86924000|86924100|86924001|86923700|86929100|86627800|86627801|86627802|86627803|86924700|86629668|86629669|86629650|86911000|86911001|86911003|86628700|86628701|86628800|86628801|86642400|86642504|86642505|86629110|86629111|86642516|86642517|86629112|86629113|86642511|86642512|86629601|86911800|86340702|86910000|86911100|86911101|86293407|86293408|86348200|86348201|86348202|86629606|86911009|86911010|86911011|86911006|86911007|86629652|86629630|86629631|86629616|86629617|86911037|86629635|86629637|86911019|86911040|86293423|86911021|86911020|86629622|86929504|86629628|86629629|86642503|86911034|86909501|86911030|86911301|86911302|86911200|86911038|86629639|86929508|86629644|86642506|86642507|86642508|86911039|86929506|86929507|86911035|86911036|86629648|86642509|86642510|86629651|86911047|86911048|86911064|86911065|86293400|86293401|86629674|86911069|86911070|86629658|86911066|86911067|86911061|86911062|86642513|86642514|86629655|86629657|86629653|86629654|86911068|86629656|86629663|86629672|86911074|86911075|86911090|86911091|86911092|86911076|86911077|86911093|86911094|86911095|86911078|86911079|86911096|86911097|86911098|86911072|86629664|86911600|86293402|86929513|86293419|86293418|86293412|86629671|86911503|86911504|86293416|86922900|86293420|86293421|86293422|86330501|86642523|86911032|86293426|86928800|86629676|86293439|86629691|86293434|86293444|86925400|86293432|86293451|86330509|86330510|86330511|86330512|86330513|86824501|86844806|86519320|86519321|86844702|86844813|86519318|86560601|86599402|86519323|86519301|86521800|86521802|86844700|86536601|86519330|86519310|86519308|86519316|86519305|86519309|86516808|86516809|86536602|86521201|86521202|86599403|86599401|86844807|86824500|86519303|86844812|86844703|86629697|86629692|86293415|86293446|86293447|86293448)\\d{8}|(868375202|868375602|863503001|861250004|861250006|861250003|861250011|861250001|861250012|861250008|861250010|861250005|861250002|861250009|861250007|861347002|861347008|861347005|861347003|861347001|861347006|861347007|861399017|861399008|861399001|861399012|861399016|861399015|861399004|861399025|861399026|861399023|861399024|861399028|861399002|861399018|861399014|861399010|861399009|861399011|861399013|861399003|861399007|861399005|861399006|861399021|861399019|861399027|861399029|861399020|861399022|861316001|861355052|861355002|861355051|861355001|861656003|861680002|861680102|861680007|861680107|861680009|861680109|861680112|861680010|861680110|861680005|861680105|861680003|861680103|861680113|861680004|861680104|861680006|861680106|861680008|861680108|861680011|861680111|863535002|863535005|863535006|863535008|863535003|863535013|863535010|863535001|863535004|863535007|863535012|863535011|863535009)\\d{7})\\b"
	re := MustCompile(pattern)

	res := re.FindAllStringIndex("hello 8614855715679961 world", -1)
	if len(res) != 1 {
		t.Errorf("expected 1 match, got %d", len(res))
	}
	if res[0][0] != 6 || res[0][1] != 22 {
		t.Errorf("expected match at 6,22, got %d,%d", res[0][0], res[0][1])
	}
}

// TestHeavyGC is a regression test for https://github.com/wasilibs/go-re2/issues/98
func TestHeavyGC(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}

	t.Parallel()

	go func() {
		ticker := time.NewTicker(time.Millisecond * 100)
		for {
			<-ticker.C
			runtime.GC()
		}
	}()
	pat := MustCompile(`(?m)(?:^|\b)(?P<pattern>[A-Za-z0-9._%+\-]+@[A-Za-z0-9.\-]+\.[A-Za-z]{2,4})(?:$|\b)`)
	for i := 0; i < 10; i++ {
		t.Run(fmt.Sprintf("test-%d", i), func(t *testing.T) {
			t.Parallel()

			body := randSeq(1024*1024*100, 0, letters)
			_ = pat.FindSubmatchIndex(body)
		})
	}
	runtime.KeepAlive(pat)
}

// nolint: gosec
func randSeq(n int, seed int64, letters []rune) []byte {
	b := make([]rune, n)
	src := rand.New(rand.NewSource(seed))
	for i := range b {
		b[i] = letters[src.Intn(len(letters))]
	}
	return []byte(string(b))
}

var letters = []rune("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ")
