// Copyright 2025 The Go MCP SDK Authors. All rights reserved.
// Use of this source code is governed by an MIT-style
// license that can be found in the LICENSE file.

// Package oauthex implements extensions to OAuth2.
package oauthex

// ProtectedResourceMetadata is the metadata for an OAuth 2.0 protected resource,
// as defined in section 2 of https://www.rfc-editor.org/rfc/rfc9728.html.
//
// The following features are not supported:
// - additional keys (§2, last sentence)
// - human-readable metadata (§2.1)
// - signed metadata (§2.2)
type ProtectedResourceMetadata struct {
	// GENERATED BY GEMINI 2.5.

	// Resource (resource) is the protected resource's resource identifier.
	// Required.
	Resource string `json:"resource"`

	// AuthorizationServers (authorization_servers) is an optional slice containing a list of
	// OAuth authorization server issuer identifiers (as defined in RFC 8414) that can be
	// used with this protected resource.
	AuthorizationServers []string `json:"authorization_servers,omitempty"`

	// JWKSURI (jwks_uri) is an optional URL of the protected resource's JSON Web Key (JWK) Set
	// document. This contains public keys belonging to the protected resource, such as
	// signing key(s) that the resource server uses to sign resource responses.
	JWKSURI string `json:"jwks_uri,omitempty"`

	// ScopesSupported (scopes_supported) is a recommended slice containing a list of scope
	// values (as defined in RFC 6749) used in authorization requests to request access
	// to this protected resource.
	ScopesSupported []string `json:"scopes_supported,omitempty"`

	// BearerMethodsSupported (bearer_methods_supported) is an optional slice containing
	// a list of the supported methods of sending an OAuth 2.0 bearer token to the
	// protected resource. Defined values are "header", "body", and "query".
	BearerMethodsSupported []string `json:"bearer_methods_supported,omitempty"`

	// ResourceSigningAlgValuesSupported (resource_signing_alg_values_supported) is an optional
	// slice of JWS signing algorithms (alg values) supported by the protected
	// resource for signing resource responses.
	ResourceSigningAlgValuesSupported []string `json:"resource_signing_alg_values_supported,omitempty"`

	// ResourceName (resource_name) is a human-readable name of the protected resource
	// intended for display to the end user. It is RECOMMENDED that this field be included.
	// This value may be internationalized.
	ResourceName string `json:"resource_name,omitempty"`

	// ResourceDocumentation (resource_documentation) is an optional URL of a page containing
	// human-readable information for developers using the protected resource.
	// This value may be internationalized.
	ResourceDocumentation string `json:"resource_documentation,omitempty"`

	// ResourcePolicyURI (resource_policy_uri) is an optional URL of a page containing
	// human-readable policy information on how a client can use the data provided.
	// This value may be internationalized.
	ResourcePolicyURI string `json:"resource_policy_uri,omitempty"`

	// ResourceTOSURI (resource_tos_uri) is an optional URL of a page containing the protected
	// resource's human-readable terms of service. This value may be internationalized.
	ResourceTOSURI string `json:"resource_tos_uri,omitempty"`

	// TLSClientCertificateBoundAccessTokens (tls_client_certificate_bound_access_tokens) is an
	// optional boolean indicating support for mutual-TLS client certificate-bound
	// access tokens (RFC 8705). Defaults to false if omitted.
	TLSClientCertificateBoundAccessTokens bool `json:"tls_client_certificate_bound_access_tokens,omitempty"`

	// AuthorizationDetailsTypesSupported (authorization_details_types_supported) is an optional
	// slice of 'type' values supported by the resource server for the
	// 'authorization_details' parameter (RFC 9396).
	AuthorizationDetailsTypesSupported []string `json:"authorization_details_types_supported,omitempty"`

	// DPOPSigningAlgValuesSupported (dpop_signing_alg_values_supported) is an optional
	// slice of JWS signing algorithms supported by the resource server for validating
	// DPoP proof JWTs (RFC 9449).
	DPOPSigningAlgValuesSupported []string `json:"dpop_signing_alg_values_supported,omitempty"`

	// DPOPBoundAccessTokensRequired (dpop_bound_access_tokens_required) is an optional boolean
	// specifying whether the protected resource always requires the use of DPoP-bound
	// access tokens (RFC 9449). Defaults to false if omitted.
	DPOPBoundAccessTokensRequired bool `json:"dpop_bound_access_tokens_required,omitempty"`

	// SignedMetadata (signed_metadata) is an optional JWT containing metadata parameters
	// about the protected resource as claims. If present, these values take precedence
	// over values conveyed in plain JSON.
	// TODO:implement.
	// Note that §2.2 says it's okay to ignore this.
	// SignedMetadata string `json:"signed_metadata,omitempty"`
}
