package squashfs

import (
	"bytes"
	"fmt"
	"reflect"
	"strings"
	"testing"
)

func TestNewCompressor(t *testing.T) {
	tests := []struct {
		flavour compression
		c       Compressor
		err     error
	}{
		{compressionGzip, &CompressorGzip{}, nil},
		{compressionLzma, &CompressorLzma{}, nil},
		{compressionLzo, nil, fmt.Errorf("LZO compression not yet supported")},
		{compressionXz, &CompressorXz{}, nil},
		{compressionLz4, &CompressorLz4{}, nil},
		{compressionZstd, &CompressorZstd{}, nil},
		{100, nil, fmt.Errorf("unknown compression type")},
	}

	for i, tt := range tests {
		c, err := newCompressor(tt.flavour)
		switch {
		case (err == nil && tt.err != nil) || (err != nil && tt.err == nil) || (err != nil && tt.err != nil && !strings.HasPrefix(err.Error(), tt.err.Error())):
			t.Errorf("%d: mismatched error, actual then expected", i)
			t.Logf("%v", err)
			t.Logf("%v", tt.err)
		case reflect.TypeOf(c) != reflect.TypeOf(tt.c):
			t.Errorf("%d: result %v did not implement %v", i, c, tt.c)
		}
	}
}

var testCompressUncompressed = []byte{
	0xde, 0xc9, 0x4e, 0xd0, 0xef, 0x19, 0xdb, 0x0a, 0x6a, 0x35, 0x26, 0x61,
	0x86, 0x2d, 0xa0, 0x42, 0x18, 0xa8, 0x89, 0xe9, 0xc4, 0x7b, 0x1a, 0xc7,
	0x85, 0x8e, 0xd6, 0x36, 0xd6, 0x83, 0x84, 0x21, 0xf4, 0x06, 0x38, 0x07,
	0x7b, 0x33, 0x3f, 0x72, 0x4c, 0xae, 0xcd, 0xfd, 0xa0, 0xb0, 0x71, 0x2f,
	0x64, 0x62, 0x62, 0x2f, 0xc3, 0x5f, 0xa1, 0x21, 0xc6, 0xbf, 0x2c, 0x39,
	0xef, 0x56, 0x23, 0x61, 0xb0, 0x98, 0x84, 0xcd, 0x24, 0xc4, 0xbf, 0x30,
	0xae, 0xd9, 0x9e, 0xb0, 0x7b, 0xc5, 0xa3, 0x8d, 0xf7, 0x4f, 0xb8, 0xdd,
	0x7b, 0x77, 0xb6, 0x8c, 0x5a, 0x10, 0xa4, 0xce, 0xc2, 0x0a, 0x3d, 0x51,
	0x23, 0x8c, 0x10, 0x1a,
}

// we cannot actually check that the compression gives the exact same output,
//  as many compressions are not completely deterministic
//  instead, we test that our decompression works, and then test that we can compress
//   followed by (already validated) decompression

//nolint:thelper // this is not a helper function
func testCompressAndDecompress(t *testing.T, c Compressor, compressed []byte) {
	t.Run("decompress", func(t *testing.T) {
		out, err := c.decompress(compressed)
		switch {
		case err != nil:
			t.Errorf("unexpected error: %v", err)
		case !bytes.Equal(out, testCompressUncompressed):
			t.Errorf("Mismatched decompressed, actual then expected")
			t.Logf("% x", out)
			t.Logf("% x", testCompressUncompressed)
		}
	})
	t.Run("compress", func(t *testing.T) {
		out, err := c.compress(testCompressUncompressed)
		if err != nil {
			t.Errorf("unexpected error: %v", err)
			return
		}
		decompressed, err := c.decompress(out)
		switch {
		case err != nil:
			t.Errorf("unexpected error: %v", err)
		case !bytes.Equal(decompressed, testCompressUncompressed):
			t.Errorf("Mismatched decompression, actual then expected")
			t.Logf("% x", decompressed)
			t.Logf("% x", testCompressUncompressed)
		}
	})
}

func TestCompressionGzip(t *testing.T) {
	compressed := []byte{
		0x78, 0x9c, 0x00, 0x64, 0x00, 0x9b, 0xff, 0xde, 0xc9, 0x4e, 0xd0, 0xef,
		0x19, 0xdb, 0x0a, 0x6a, 0x35, 0x26, 0x61, 0x86, 0x2d, 0xa0, 0x42, 0x18,
		0xa8, 0x89, 0xe9, 0xc4, 0x7b, 0x1a, 0xc7, 0x85, 0x8e, 0xd6, 0x36, 0xd6,
		0x83, 0x84, 0x21, 0xf4, 0x06, 0x38, 0x07, 0x7b, 0x33, 0x3f, 0x72, 0x4c,
		0xae, 0xcd, 0xfd, 0xa0, 0xb0, 0x71, 0x2f, 0x64, 0x62, 0x62, 0x2f, 0xc3,
		0x5f, 0xa1, 0x21, 0xc6, 0xbf, 0x2c, 0x39, 0xef, 0x56, 0x23, 0x61, 0xb0,
		0x98, 0x84, 0xcd, 0x24, 0xc4, 0xbf, 0x30, 0xae, 0xd9, 0x9e, 0xb0, 0x7b,
		0xc5, 0xa3, 0x8d, 0xf7, 0x4f, 0xb8, 0xdd, 0x7b, 0x77, 0xb6, 0x8c, 0x5a,
		0x10, 0xa4, 0xce, 0xc2, 0x0a, 0x3d, 0x51, 0x23, 0x8c, 0x10, 0x1a, 0x01,
		0x00, 0x00, 0xff, 0xff, 0xaf, 0x57, 0x30, 0xea,
	}

	c := CompressorGzip{CompressionLevel: 2}
	testCompressAndDecompress(t, &c, compressed)
}
func TestCompressionLzma(t *testing.T) {
	compressed := []byte{
		0x5d, 0x00, 0x00, 0x80, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0x00, 0x6f, 0x32, 0x45, 0x7e, 0x9f, 0x76, 0xce, 0x6a, 0x49, 0xc5,
		0x98, 0x49, 0x83, 0x1a, 0x4f, 0x0d, 0x4e, 0x46, 0x06, 0xfa, 0xd2, 0xb4,
		0xaa, 0x48, 0x6e, 0x81, 0x1e, 0xcd, 0x23, 0x3a, 0xc8, 0xaf, 0xdd, 0xc9,
		0xfd, 0xa8, 0xbe, 0x67, 0xd1, 0x69, 0xd8, 0x0c, 0x84, 0xb9, 0xa0, 0x34,
		0xb9, 0xa2, 0xfe, 0x0f, 0xdc, 0x9c, 0x8f, 0x17, 0x26, 0xb2, 0xae, 0x7b,
		0x01, 0xe7, 0x1f, 0x7c, 0xec, 0xfd, 0xd6, 0xdf, 0x23, 0x0c, 0x53, 0xf5,
		0x94, 0xb1, 0x31, 0x84, 0x66, 0x8b, 0xf5, 0x88, 0xdd, 0xa2, 0x69, 0x1a,
		0xeb, 0x35, 0xbd, 0xb5, 0x59, 0x83, 0x93, 0xb5, 0xfa, 0x65, 0x74, 0x25,
		0x19, 0x4e, 0x90, 0x8d, 0xd2, 0x70, 0xac, 0x48, 0x23, 0xb6, 0x0a, 0xe4,
		0x99, 0x88, 0xf3, 0x0e, 0xf6, 0xff, 0xff, 0xe2, 0xeb, 0x00, 0x00,
	}
	c := CompressorLzma{}
	testCompressAndDecompress(t, &c, compressed)
}
func TestCompressionXz(t *testing.T) {
	compressed := []byte{
		0xfd, 0x37, 0x7a, 0x58, 0x5a, 0x00, 0x00, 0x04, 0xe6, 0xd6, 0xb4, 0x46,
		0x02, 0x00, 0x21, 0x01, 0x16, 0x00, 0x00, 0x00, 0x74, 0x2f, 0xe5, 0xa3,
		0x01, 0x00, 0x63, 0xde, 0xc9, 0x4e, 0xd0, 0xef, 0x19, 0xdb, 0x0a, 0x6a,
		0x35, 0x26, 0x61, 0x86, 0x2d, 0xa0, 0x42, 0x18, 0xa8, 0x89, 0xe9, 0xc4,
		0x7b, 0x1a, 0xc7, 0x85, 0x8e, 0xd6, 0x36, 0xd6, 0x83, 0x84, 0x21, 0xf4,
		0x06, 0x38, 0x07, 0x7b, 0x33, 0x3f, 0x72, 0x4c, 0xae, 0xcd, 0xfd, 0xa0,
		0xb0, 0x71, 0x2f, 0x64, 0x62, 0x62, 0x2f, 0xc3, 0x5f, 0xa1, 0x21, 0xc6,
		0xbf, 0x2c, 0x39, 0xef, 0x56, 0x23, 0x61, 0xb0, 0x98, 0x84, 0xcd, 0x24,
		0xc4, 0xbf, 0x30, 0xae, 0xd9, 0x9e, 0xb0, 0x7b, 0xc5, 0xa3, 0x8d, 0xf7,
		0x4f, 0xb8, 0xdd, 0x7b, 0x77, 0xb6, 0x8c, 0x5a, 0x10, 0xa4, 0xce, 0xc2,
		0x0a, 0x3d, 0x51, 0x23, 0x8c, 0x10, 0x1a, 0x00, 0x76, 0x9b, 0x80, 0xe2,
		0x55, 0x86, 0x1c, 0x16, 0x00, 0x01, 0x7c, 0x64, 0x90, 0x26, 0xd3, 0xe9,
		0x1f, 0xb6, 0xf3, 0x7d, 0x01, 0x00, 0x00, 0x00, 0x00, 0x04, 0x59, 0x5a,
	}
	c := CompressorXz{}
	testCompressAndDecompress(t, &c, compressed)
}
func TestCompressionLz4(t *testing.T) {
	compressed := []byte{
		0x04, 0x22, 0x4d, 0x18, 0x64, 0x40, 0xa7, 0x64, 0x00, 0x00, 0x80, 0xde,
		0xc9, 0x4e, 0xd0, 0xef, 0x19, 0xdb, 0x0a, 0x6a, 0x35, 0x26, 0x61, 0x86,
		0x2d, 0xa0, 0x42, 0x18, 0xa8, 0x89, 0xe9, 0xc4, 0x7b, 0x1a, 0xc7, 0x85,
		0x8e, 0xd6, 0x36, 0xd6, 0x83, 0x84, 0x21, 0xf4, 0x06, 0x38, 0x07, 0x7b,
		0x33, 0x3f, 0x72, 0x4c, 0xae, 0xcd, 0xfd, 0xa0, 0xb0, 0x71, 0x2f, 0x64,
		0x62, 0x62, 0x2f, 0xc3, 0x5f, 0xa1, 0x21, 0xc6, 0xbf, 0x2c, 0x39, 0xef,
		0x56, 0x23, 0x61, 0xb0, 0x98, 0x84, 0xcd, 0x24, 0xc4, 0xbf, 0x30, 0xae,
		0xd9, 0x9e, 0xb0, 0x7b, 0xc5, 0xa3, 0x8d, 0xf7, 0x4f, 0xb8, 0xdd, 0x7b,
		0x77, 0xb6, 0x8c, 0x5a, 0x10, 0xa4, 0xce, 0xc2, 0x0a, 0x3d, 0x51, 0x23,
		0x8c, 0x10, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x0c, 0xda, 0x4b, 0x2b,
	}
	c := CompressorLz4{}
	testCompressAndDecompress(t, &c, compressed)
}
func TestCompressionZstd(t *testing.T) {
	compressed := []byte{
		0x28, 0xb5, 0x2f, 0xfd, 0x04, 0x00, 0x21, 0x03, 0x00, 0xde, 0xc9, 0x4e,
		0xd0, 0xef, 0x19, 0xdb, 0x0a, 0x6a, 0x35, 0x26, 0x61, 0x86, 0x2d, 0xa0,
		0x42, 0x18, 0xa8, 0x89, 0xe9, 0xc4, 0x7b, 0x1a, 0xc7, 0x85, 0x8e, 0xd6,
		0x36, 0xd6, 0x83, 0x84, 0x21, 0xf4, 0x06, 0x38, 0x07, 0x7b, 0x33, 0x3f,
		0x72, 0x4c, 0xae, 0xcd, 0xfd, 0xa0, 0xb0, 0x71, 0x2f, 0x64, 0x62, 0x62,
		0x2f, 0xc3, 0x5f, 0xa1, 0x21, 0xc6, 0xbf, 0x2c, 0x39, 0xef, 0x56, 0x23,
		0x61, 0xb0, 0x98, 0x84, 0xcd, 0x24, 0xc4, 0xbf, 0x30, 0xae, 0xd9, 0x9e,
		0xb0, 0x7b, 0xc5, 0xa3, 0x8d, 0xf7, 0x4f, 0xb8, 0xdd, 0x7b, 0x77, 0xb6,
		0x8c, 0x5a, 0x10, 0xa4, 0xce, 0xc2, 0x0a, 0x3d, 0x51, 0x23, 0x8c, 0x10,
		0x1a, 0x18, 0x20, 0x19, 0x3b,
	}
	c := CompressorZstd{}
	testCompressAndDecompress(t, &c, compressed)
}
