package io

import (
	"bytes"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"sort"
	"strings"

	"git.sr.ht/~charles/rq/util"
)

var inputHandlers map[string]func() InputHandler

// InputHandler represents a handler for one type of input format, such as JSON
// or CSV.
type InputHandler interface {

	// Parse consumes input from a reader and unmarshals it into an
	// interface. This will later be passed to ast.InterfaceToValue()
	// for consumption by Rego.
	Parse(reader io.Reader) (interface{}, error)

	// SetOption specifies an option to control how the handler should
	// behave. Setting options that the handler does not implement should
	// cause SetOption to return nil, this way the CLI handling logic does
	// not have to determine which handler is being used before deciding
	// which options to try to set.
	//
	// The canonical method for parsing non-string values is to use
	// util.StringToValue().
	SetOption(name string, value string) error

	// Name should return the name of this handler, e.g. "json" or "csv".
	Name() string
}

// SelectInputHandler chooses a handler based on the name provided.
//
// If name is the empty string, then it default to "json".
func SelectInputHandler(name string) (InputHandler, error) {
	if name == "" {
		name = "json"
	}

	h, ok := inputHandlers[name]
	if !ok {
		return nil, fmt.Errorf("no such handler with name '%s'", name)
	}

	return h(), nil
}

// ListInputHandlers lists all known handler names.
func ListInputHandlers() []string {
	l := []string{}
	for h := range inputHandlers {
		l = append(l, h)
	}
	sort.Strings(l)
	return l
}

func registerInputHandler(name string, newFunc func() InputHandler) {
	if inputHandlers == nil {
		inputHandlers = make(map[string]func() InputHandler)
	}

	inputHandlers[name] = newFunc
}

// LoadInput loads the input file specified by the DataSpec to an interface.
//
// If the DataSpec does not specify a format (format is the empty string), the
// format defaults to the file extension if it is a known format, and else
// "json".
//
// If the FilePath is the empty string, then an empty buffer is used as the
// input file to Parse(). When this happens, the option
// `generic.empty_file_path` will be set to true.
func LoadInput(d *DataSpec) (interface{}, error) {
	// Resolve the reader to either a file descriptor or an empty buffer
	// if the FilePath is empty.
	var reader io.Reader = &bytes.Buffer{}
	if d.FilePath != "" {
		// Note that setting generic.empty_file_path happens in
		// LoadInputFromReader.
		f, err := os.Open(d.FilePath)
		if err != nil {
			return nil, err
		}
		defer func() { _ = f.Close() }()
		reader = f
	}

	return LoadInputFromReader(d, reader)
}

// LoadInputFromReader behaves like LoadInput, except that it ignores the
// FilePath in the DataSpec, and instead reads input from the provided reader.
//
// Like with LoadInput, if the FilePath is omitted, then
// generic.empty_file_path is set to true.
func LoadInputFromReader(d *DataSpec, r io.Reader) (interface{}, error) {

	// Resolve the format to the given format, the extension, or JSON.
	format := d.Format
	if format == "" {
		format = strings.ToLower(filepath.Ext(d.FilePath))
	}

	handler, err := SelectInputHandler(format)
	if err != nil {
		handler, err = SelectInputHandler("json")
		if err != nil {
			// we should always be able to select the JSON handler
			panic(err)
		}
	}

	if d.Options != nil {
		for o, v := range d.Options {
			if err := handler.SetOption(o, v); err != nil {
				return nil, err
			}
		}
	}

	if d.FilePath == "" {
		err := handler.SetOption("generic.empty_file_path", "true")
		if err != nil {
			return nil, err
		}
	}

	// we should treat input errors strictly by default
	strict := true
	if strictS, ok := d.Options["strict"]; ok {
		strict = util.Truthy(strictS)
	}

	result, err := handler.Parse(r)
	if (err != nil) && (!strict) {
		return nil, nil
	}

	return result, err
}
