// Copyright The OpenTelemetry Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package featuregate // import "go.opentelemetry.io/collector/service/featuregate"

import (
	"fmt"
	"sync"
)

// Gate represents an individual feature that may be enabled or disabled based
// on the lifecycle state of the feature and CLI flags specified by the user.
type Gate struct {
	ID          string
	Description string
	Enabled     bool
}

var reg = NewRegistry()

// GetRegistry returns the global Registry.
func GetRegistry() *Registry {
	return reg
}

// Deprecated: [v0.50.0] Use GetRegistry().Apply.
var Apply = GetRegistry().Apply

// Deprecated: [v0.50.0] Use GetRegistry().IsEnabled.
var IsEnabled = GetRegistry().IsEnabled

// Deprecated: [v0.50.0] Use GetRegistry().List.
var List = GetRegistry().List

// Deprecated: [v0.50.0] Use GetRegistry().MustRegister.
var Register = GetRegistry().MustRegister

// NewRegistry returns a new empty Registry.
func NewRegistry() *Registry {
	return &Registry{gates: make(map[string]Gate)}
}

type Registry struct {
	mu    sync.RWMutex
	gates map[string]Gate
}

// Apply a configuration in the form of a map of Gate identifiers to boolean values.
// Sets only those values provided in the map, other gate values are not changed.
func (r *Registry) Apply(cfg map[string]bool) {
	r.mu.Lock()
	defer r.mu.Unlock()
	for id, val := range cfg {
		if g, ok := r.gates[id]; ok {
			g.Enabled = val
			r.gates[g.ID] = g
		}
	}
}

// IsEnabled returns true if a registered feature gate is enabled and false otherwise.
func (r *Registry) IsEnabled(id string) bool {
	r.mu.RLock()
	defer r.mu.RUnlock()
	g, ok := r.gates[id]
	return ok && g.Enabled
}

// MustRegister like Register but panics if a Gate with the same ID is already registered.
func (r *Registry) MustRegister(g Gate) {
	if err := r.Register(g); err != nil {
		panic(err)
	}
}

// Register registers a Gate. May only be called in an init() function.
func (r *Registry) Register(g Gate) error {
	r.mu.Lock()
	defer r.mu.Unlock()
	if _, ok := r.gates[g.ID]; ok {
		return fmt.Errorf("attempted to add pre-existing gate %q", g.ID)
	}
	r.gates[g.ID] = g
	return nil
}

// List returns a slice of copies of all registered Gates.
func (r *Registry) List() []Gate {
	r.mu.RLock()
	defer r.mu.RUnlock()
	ret := make([]Gate, len(r.gates))
	i := 0
	for _, gate := range r.gates {
		ret[i] = gate
		i++
	}

	return ret
}
