import { Browser, Builder, Capabilities } from "selenium-webdriver";
import { Options, ServiceBuilder } from "selenium-webdriver/chrome";
import { existsSync, mkdirSync } from "fs";
import ErrorProcessor from "./ErrorProcessor";
import { resolve } from "path";
export default class Driver {
    static async init() {
        const chromeExtensionPath = process.env.UNZIPPED_CHROME_EXTENSION_PATH
            ? resolve(process.env.UNZIPPED_CHROME_EXTENSION_PATH)
            : resolve("dist");
        if (!existsSync(chromeExtensionPath)) {
            throw new Error("Please set UNZIPPED_CHROME_EXTENSION_PATH variable to unzipped Chrome extension directory. " +
                "For example: export UNZIPPED_CHROME_EXTENSION_PATH=/path/to/dist. " +
                "Directory " +
                chromeExtensionPath +
                " does not exist.");
        }
        const CHROME_DIR = resolve("dist-tests-e2e", "chrome_data");
        if (!existsSync(CHROME_DIR)) {
            mkdirSync(CHROME_DIR, { recursive: true });
        }
        const chromeOptions = new Options();
        chromeOptions.addArguments("--user-data-dir=" + CHROME_DIR, "--load-extension=" + chromeExtensionPath, "--enable-features=UnexpireFlagsM118", "--allow-insecure-localhost", "--disable-web-security", "--remote-allow-origins=*", "--disable-gpu");
        const LOGS_DIR = resolve("dist-tests-e2e", "logs");
        if (!existsSync(LOGS_DIR)) {
            mkdirSync(LOGS_DIR, { recursive: true });
        }
        const chromeServiceBuilder = new ServiceBuilder();
        chromeServiceBuilder.loggingTo(LOGS_DIR + "/chromedriver.log").enableVerboseLogging();
        const driver = await new Builder()
            .withCapabilities(Capabilities.chrome())
            .setChromeService(chromeServiceBuilder)
            .forBrowser(Browser.CHROME)
            .setChromeOptions(chromeOptions)
            .build();
        await ErrorProcessor.run(async () => await driver.manage().window().maximize(), "Error while maximizing browser window.");
        return driver;
    }
    static async openUrl(driver, url) {
        return await ErrorProcessor.run(async () => await driver.get(url), "Error while opening url: " + url);
    }
    static async quit(driver) {
        return await ErrorProcessor.run(async () => await driver.quit(), "Error while quiting driver.");
    }
}
//# sourceMappingURL=Driver.js.map