import * as RF from "reactflow";
import { useState, useCallback, useMemo } from "react";
import { addEdgeWaypoint } from "../../mutations/addEdgeWaypoint";
import { useDmnEditorStore, useDmnEditorStoreApi } from "../../store/StoreContext";
import { snapPoint } from "../SnapGrid";
import { xmlHrefToQName } from "@kie-tools/dmn-marshaller/dist/xml/xmlHrefToQName";
import { useExternalModels } from "../../includedModels/DmnEditorDependenciesContext";
import { addEdge } from "../../mutations/addEdge";
import { getHandlePosition } from "../maths/DmnMaths";
export function usePotentialWaypointControls(waypoints, isEdgeSelected, edgeId, edgeIndex, interactionPathRef) {
    const snapGrid = useDmnEditorStore((s) => s.diagram.snapGrid);
    const drdIndex = useDmnEditorStore((s) => s.computed(s).getDrdIndex());
    const isDraggingWaypoint = useDmnEditorStore((s) => !!s.diagram.draggingWaypoints.find((e) => e === edgeId));
    const dmnEditorStoreApi = useDmnEditorStoreApi();
    const reactFlowInstance = RF.useReactFlow();
    const { externalModelsByNamespace } = useExternalModels();
    const [potentialWaypoint, setPotentialWaypoint] = useState(undefined);
    const isConnecting = !!RF.useStore((s) => s.connectionNodeId);
    const isExistingWaypoint = useCallback((point) => waypoints.find((w) => w["@_x"] === point["@_x"] && w["@_y"] === point["@_y"]), [waypoints]);
    const onMouseMove = useCallback((e) => {
        const projectedPoint = reactFlowInstance.screenToFlowPosition({
            x: e.clientX,
            y: e.clientY,
        });
        setPotentialWaypoint(approximateClosestPoint(interactionPathRef.current, [projectedPoint.x, projectedPoint.y]));
    }, [interactionPathRef, reactFlowInstance]);
    const snappedPotentialWaypoint = useMemo(() => {
        if (!potentialWaypoint) {
            return undefined;
        }
        return snapPoint(snapGrid, {
            "@_x": potentialWaypoint.point.x,
            "@_y": potentialWaypoint.point.y,
        });
    }, [snapGrid, potentialWaypoint]);
    const onDoubleClick = useCallback(() => {
        if (!potentialWaypoint || !snappedPotentialWaypoint) {
            return;
        }
        if (edgeIndex === undefined) {
            dmnEditorStoreApi.setState((state) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const nodesById = state.computed(state).getDiagramData(externalModelsByNamespace).nodesById;
                const edge = state.computed(state).getDiagramData(externalModelsByNamespace).edgesById.get(edgeId);
                if (edge === undefined || ((_a = edge.data) === null || _a === void 0 ? void 0 : _a.dmnShapeSource) === undefined || ((_b = edge.data) === null || _b === void 0 ? void 0 : _b.dmnShapeTarget) == undefined) {
                    console.debug(`DMN DIAGRAM: We can not add DMNEdge for '${edgeId}' edge into diagram. There are missing data edge: ${edge}, edge.data: ${edge === null || edge === void 0 ? void 0 : edge.data}`);
                    return;
                }
                const edgeSourceBounds = (_c = edge.data) === null || _c === void 0 ? void 0 : _c.dmnShapeSource["dc:Bounds"];
                const edgeTargetBounds = (_d = edge.data) === null || _d === void 0 ? void 0 : _d.dmnShapeTarget["dc:Bounds"];
                if (edgeSourceBounds === undefined || edgeTargetBounds === undefined) {
                    console.debug(`DMN DIAGRAM: We can not add DMNEdge for '${edgeId}' edge into diagram. There are missing data edgeSourceBounds: ${edgeSourceBounds}, edgeTargetBounds: ${edgeTargetBounds}`);
                    return;
                }
                const sourceNode = nodesById.get(edge.source);
                const targetNode = nodesById.get(edge.target);
                if (sourceNode === undefined || targetNode === undefined) {
                    console.debug(`DMN DIAGRAM: We can not add DMNEdge for '${edgeId}' edge into diagram. There are missing data sourceNode: ${sourceNode}, targetNode: ${targetNode}`);
                    return;
                }
                const targetsExternalNode = targetNode.data.dmnObjectQName.prefix !== undefined;
                const requirementEdgeQNameRelativeToThisDmn = xmlHrefToQName(edgeId, state.dmn.model.definitions);
                addEdge({
                    definitions: state.dmn.model.definitions,
                    drdIndex: state.computed(state).getDrdIndex(),
                    edge: {
                        type: edge.type,
                        targetHandle: getHandlePosition({ shapeBounds: edgeTargetBounds, waypoint: snappedPotentialWaypoint })
                            .handlePosition,
                        sourceHandle: getHandlePosition({ shapeBounds: edgeSourceBounds, waypoint: snappedPotentialWaypoint })
                            .handlePosition,
                        autoPositionedEdgeMarker: undefined,
                    },
                    sourceNode: {
                        type: sourceNode.type,
                        data: sourceNode.data,
                        href: edge.source,
                        bounds: edgeSourceBounds,
                        shapeId: (_e = edge.data) === null || _e === void 0 ? void 0 : _e.dmnShapeSource["@_id"],
                    },
                    targetNode: {
                        type: targetNode.type,
                        href: edge.target,
                        data: targetNode.data,
                        bounds: edgeTargetBounds,
                        index: (_g = (_f = nodesById.get(edge.target)) === null || _f === void 0 ? void 0 : _f.data.index) !== null && _g !== void 0 ? _g : 0,
                        shapeId: (_h = edge.data) === null || _h === void 0 ? void 0 : _h.dmnShapeTarget["@_id"],
                    },
                    keepWaypoints: false,
                    externalModelsByNamespace,
                    dmnElementRefOfDmnEdge: targetsExternalNode ? requirementEdgeQNameRelativeToThisDmn : undefined,
                });
                console.debug(`DMN DIAGRAM: DMNEdge for '${edgeId}' edge was added into diagram.`);
            });
        }
        if (isExistingWaypoint(snappedPotentialWaypoint)) {
            console.debug("DMN DIAGRAM: Preventing overlapping waypoint creation.");
            return;
        }
        let i = 1;
        for (let currentLength = 0; currentLength < potentialWaypoint.lengthInPath; i++) {
            currentLength += Math.sqrt(distanceComponentsSquared([waypoints[i]["@_x"], waypoints[i]["@_y"]], {
                x: waypoints[i - 1]["@_x"],
                y: waypoints[i - 1]["@_y"],
            }));
        }
        dmnEditorStoreApi.setState((state) => {
            var _a;
            const edgeQName = xmlHrefToQName(edgeId, state.dmn.model.definitions);
            const dmnEdgeIndex = (_a = state.computed(state).indexedDrd().dmnEdgesByDmnElementRef.get(edgeQName)) === null || _a === void 0 ? void 0 : _a.index;
            if (dmnEdgeIndex === undefined) {
                throw new Error(`DMN DIAGRAM: Diagram computed state does not contain DMNEdge for '${edgeId}' edge.`);
            }
            addEdgeWaypoint({
                definitions: state.dmn.model.definitions,
                drdIndex,
                beforeIndex: i - 1,
                dmnEdgeIndex,
                waypoint: snappedPotentialWaypoint,
            });
            console.debug(`DMN DIAGRAM: Waypoint on the DMNEdge for '${edgeId}' edge was added.`);
        });
    }, [
        drdIndex,
        dmnEditorStoreApi,
        edgeId,
        edgeIndex,
        externalModelsByNamespace,
        isExistingWaypoint,
        potentialWaypoint,
        snappedPotentialWaypoint,
        waypoints,
    ]);
    const shouldReturnPotentialWaypoint = isEdgeSelected &&
        !isDraggingWaypoint &&
        snappedPotentialWaypoint &&
        !isExistingWaypoint(snappedPotentialWaypoint) &&
        !isConnecting;
    return {
        isDraggingWaypoint,
        onMouseMove,
        onDoubleClick,
        potentialWaypoint: !shouldReturnPotentialWaypoint ? undefined : potentialWaypoint,
    };
}
function approximateClosestPoint(pathNode, point) {
    const pathLength = pathNode.getTotalLength();
    let precision = Math.floor(pathLength / 10);
    let best;
    let bestLength = 0;
    let bestDistance = Infinity;
    let scan;
    let scanDistance;
    for (let scanLength = 0; scanLength <= pathLength; scanLength += precision) {
        scan = pathNode.getPointAtLength(scanLength);
        scanDistance = distanceComponentsSquared(point, scan);
        if (scanDistance < bestDistance) {
            best = scan;
            bestLength = scanLength;
            bestDistance = scanDistance;
        }
    }
    precision /= 2;
    while (precision > 1) {
        const bLength = bestLength - precision;
        const b = pathNode.getPointAtLength(bLength);
        const bDistance = distanceComponentsSquared(point, b);
        if (bLength >= 0 && bDistance < bestDistance) {
            best = b;
            bestLength = bLength;
            bestDistance = bDistance;
            continue;
        }
        const aLength = bestLength + precision;
        const a = pathNode.getPointAtLength(aLength);
        const aDistance = distanceComponentsSquared(point, a);
        if (aLength <= pathLength && aDistance < bestDistance) {
            best = a;
            bestLength = aLength;
            bestDistance = aDistance;
            continue;
        }
        precision /= 2;
    }
    return { point: best, lengthInPath: bestLength };
}
function distanceComponentsSquared(a, b) {
    const dx = b.x - a[0];
    const dy = b.y - a[1];
    return dx * dx + dy * dy;
}
//# sourceMappingURL=usePotentialWaypointControls.js.map