/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.configuration.util;

import static org.apache.ignite.internal.configuration.util.ConfigurationUtil.innerNodeVisitor;
import static org.apache.ignite.internal.configuration.util.ConfigurationUtil.leafNodeVisitor;
import static org.apache.ignite.internal.configuration.util.ConfigurationUtil.namedListNodeVisitor;
import static org.apache.ignite.internal.util.CollectionUtils.viewReadOnly;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.ignite.configuration.ConfigurationProperty;
import org.apache.ignite.configuration.NamedListView;
import org.apache.ignite.configuration.annotation.PolymorphicConfigInstance;
import org.apache.ignite.configuration.notifications.ConfigurationListener;
import org.apache.ignite.configuration.notifications.ConfigurationNamedListListener;
import org.apache.ignite.configuration.notifications.ConfigurationNotificationEvent;
import org.apache.ignite.internal.configuration.ConfigurationNode;
import org.apache.ignite.internal.configuration.DynamicConfiguration;
import org.apache.ignite.internal.configuration.DynamicProperty;
import org.apache.ignite.internal.configuration.NamedListConfiguration;
import org.apache.ignite.internal.configuration.tree.ConfigurationVisitor;
import org.apache.ignite.internal.configuration.tree.InnerNode;
import org.apache.ignite.internal.configuration.tree.NamedListNode;
import org.jetbrains.annotations.Nullable;

/**
 * Utility class for notifying a configuration update.
 *
 * @implNote When notifying listeners for "any" properties of a given configuration there's a caveat for Polymorphic Configuration
 *     subclasses ({@link PolymorphicConfigInstance}): their "any" configuration stub will only have properties of the superclass,
 *     because it is not possible to know beforehand the concrete implementation of a polymorphic configuration.
 *     This means that for Polymorphic Configuration subclasses, "any" listeners will only be notified for changes of keys declared
 *     in the parent superclass.
 */
// FIXME: https://issues.apache.org/jira/browse/IGNITE-15916 remove the Polymorphic Configuration constraint.
public class ConfigurationNotificationsUtil {
    /**
     * Recursively notifies all public configuration listeners, accumulating resulting futures in {@code futures} list.
     *
     * @param oldInnerNode Old configuration values root.
     * @param newInnerNode New configuration values root.
     * @param cfgNode Public configuration tree node corresponding to the current inner nodes.
     * @param storageRevision Storage revision.
     * @param futures Write-only list of futures to accumulate results.
     */
    public static void notifyListeners(
            @Nullable InnerNode oldInnerNode,
            InnerNode newInnerNode,
            DynamicConfiguration<InnerNode, ?> cfgNode,
            long storageRevision,
            List<CompletableFuture<?>> futures
    ) {
        notifyListeners(oldInnerNode, newInnerNode, cfgNode, storageRevision, futures, List.of(), new HashMap<>(), true);
    }

    /**
     * Recursively notifies all public configuration listeners, accumulating resulting futures in {@code futures} list.
     *
     * @param oldInnerNode Old configuration values root.
     * @param newInnerNode New configuration values root.
     * @param cfgNode Public configuration tree node corresponding to the current inner nodes.
     * @param storageRevision Storage revision.
     * @param futures Write-only list of futures to accumulate results.
     * @param anyConfigs Current {@link NamedListConfiguration#any "any"} configurations.
     * @param eventConfigs Configuration containers for {@link ConfigurationNotificationEvent}.
     * @param putConfigContainer Put new configuration container to {@code eventConfigs}.
     */
    private static void notifyListeners(
            @Nullable InnerNode oldInnerNode,
            InnerNode newInnerNode,
            DynamicConfiguration<InnerNode, ?> cfgNode,
            long storageRevision,
            List<CompletableFuture<?>> futures,
            Collection<DynamicConfiguration<InnerNode, ?>> anyConfigs,
            Map<Class<?>, ConfigurationContainer> eventConfigs,
            boolean putConfigContainer
    ) {
        assert !(cfgNode instanceof NamedListConfiguration);

        if (oldInnerNode == null || oldInnerNode == newInnerNode) {
            return;
        }

        if (putConfigContainer) {
            putConfigContainer(eventConfigs, cfgNode, null);
        }

        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
            notifyPublicListeners(
                    anyConfig.listeners(),
                    oldInnerNode,
                    newInnerNode,
                    storageRevision,
                    futures,
                    ConfigurationListener::onUpdate,
                    eventConfigs
            );
        }

        notifyPublicListeners(
                cfgNode.listeners(),
                oldInnerNode,
                newInnerNode,
                storageRevision,
                futures,
                ConfigurationListener::onUpdate,
                eventConfigs
        );

        // Polymorphic configuration type has changed.
        // At the moment, we do not separate common fields from fields of a specific polymorphic configuration,
        // so this may cause errors in the logic below, perhaps we will fix it later.
        if (oldInnerNode.schemaType() != newInnerNode.schemaType()) {
            return;
        }

        oldInnerNode.traverseChildren(new ConfigurationVisitor<Void>() {
            /** {@inheritDoc} */
            @Override
            public Void visitLeafNode(String key, Serializable oldLeaf) {
                Serializable newLeaf = newInnerNode.traverseChild(key, leafNodeVisitor(), true);

                if (newLeaf != oldLeaf) {
                    for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                        notifyPublicListeners(
                                listeners(dynamicProperty(anyConfig, key)),
                                oldLeaf,
                                newLeaf,
                                storageRevision,
                                futures,
                                ConfigurationListener::onUpdate,
                                eventConfigs
                        );
                    }

                    notifyPublicListeners(
                            dynamicProperty(cfgNode, key).listeners(),
                            oldLeaf,
                            newLeaf,
                            storageRevision,
                            futures,
                            ConfigurationListener::onUpdate,
                            eventConfigs
                    );
                }

                return null;
            }

            /** {@inheritDoc} */
            @Override
            public Void visitInnerNode(String key, InnerNode oldNode) {
                InnerNode newNode = newInnerNode.traverseChild(key, innerNodeVisitor(), true);

                notifyListeners(
                        oldNode,
                        newNode,
                        dynamicConfig(cfgNode, key),
                        storageRevision,
                        futures,
                        viewReadOnly(anyConfigs, cfg -> dynamicConfig(cfg, key)),
                        eventConfigs,
                        true
                );

                return null;
            }

            /** {@inheritDoc} */
            @Override
            public Void visitNamedListNode(String key, NamedListNode<?> oldNamedList) {
                var newNamedList = (NamedListNode<InnerNode>) newInnerNode.traverseChild(key, namedListNodeVisitor(), true);

                if (newNamedList != oldNamedList) {
                    for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                        notifyPublicListeners(
                                listeners(namedDynamicConfig(anyConfig, key)),
                                (NamedListView<InnerNode>) oldNamedList,
                                newNamedList,
                                storageRevision,
                                futures,
                                ConfigurationListener::onUpdate,
                                eventConfigs
                        );
                    }

                    notifyPublicListeners(
                            namedDynamicConfig(cfgNode, key).listeners(),
                            (NamedListView<InnerNode>) oldNamedList,
                            newNamedList,
                            storageRevision,
                            futures,
                            ConfigurationListener::onUpdate,
                            eventConfigs
                    );

                    // This is optimization, we could use "NamedListConfiguration#get" directly, but we don't want to.
                    List<String> oldNames = oldNamedList.namedListKeys();
                    List<String> newNames = newNamedList.namedListKeys();

                    NamedListConfiguration<?, InnerNode, ?> namedListCfg = namedDynamicConfig(cfgNode, key);

                    Map<String, ConfigurationProperty<?>> namedListCfgMembers = namedListCfg.touchMembers();

                    Set<String> created = new HashSet<>(newNames);
                    created.removeAll(oldNames);

                    Set<String> deleted = new HashSet<>(oldNames);
                    deleted.removeAll(newNames);

                    Map<String, String> renamed = new HashMap<>();
                    if (!created.isEmpty() && !deleted.isEmpty()) {
                        Map<UUID, String> createdIds = new HashMap<>();

                        for (String createdKey : created) {
                            createdIds.put(newNamedList.internalId(createdKey), createdKey);
                        }

                        // Avoiding ConcurrentModificationException.
                        for (String deletedKey : Set.copyOf(deleted)) {
                            UUID internalId = oldNamedList.internalId(deletedKey);

                            String maybeRenamedKey = createdIds.get(internalId);

                            if (maybeRenamedKey == null) {
                                continue;
                            }

                            deleted.remove(deletedKey);
                            created.remove(maybeRenamedKey);
                            renamed.put(deletedKey, maybeRenamedKey);
                        }
                    }

                    // Lazy initialization.
                    Collection<DynamicConfiguration<InnerNode, ?>> newAnyConfigs = null;

                    for (String name : created) {
                        DynamicConfiguration<InnerNode, ?> newNodeCfg =
                                (DynamicConfiguration<InnerNode, ?>) namedListCfg.members().get(name);

                        touch(newNodeCfg);

                        putConfigContainer(eventConfigs, newNodeCfg, name);

                        InnerNode newVal = newNamedList.getInnerNode(name);

                        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                            notifyPublicListeners(
                                    extendedListeners(namedDynamicConfig(anyConfig, key)),
                                    null,
                                    newVal,
                                    storageRevision,
                                    futures,
                                    ConfigurationNamedListListener::onCreate,
                                    eventConfigs
                            );
                        }

                        notifyPublicListeners(
                                namedDynamicConfig(cfgNode, key).extendedListeners(),
                                null,
                                newVal,
                                storageRevision,
                                futures,
                                ConfigurationNamedListListener::onCreate,
                                eventConfigs
                        );

                        if (newAnyConfigs == null) {
                            newAnyConfigs = mergeAnyConfigs(
                                    viewReadOnly(anyConfigs, cfg -> any(namedDynamicConfig(cfg, key))),
                                    any(namedListCfg)
                            );
                        }

                        notifyAnyListenersOnCreate(
                                newVal,
                                newNodeCfg,
                                storageRevision,
                                futures,
                                newAnyConfigs,
                                eventConfigs
                        );

                        removeConfigContainer(eventConfigs, newNodeCfg);
                    }

                    for (String name : deleted) {
                        DynamicConfiguration<InnerNode, ?> delNodeCfg =
                                (DynamicConfiguration<InnerNode, ?>) namedListCfgMembers.get(name);

                        delNodeCfg.removedFromNamedList();

                        putConfigContainer(eventConfigs, delNodeCfg, name);

                        InnerNode oldVal = oldNamedList.getInnerNode(name);

                        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                            notifyPublicListeners(
                                    extendedListeners(namedDynamicConfig(anyConfig, key)),
                                    oldVal,
                                    null,
                                    storageRevision,
                                    futures,
                                    ConfigurationNamedListListener::onDelete,
                                    eventConfigs
                            );
                        }

                        notifyPublicListeners(
                                namedDynamicConfig(cfgNode, key).extendedListeners(),
                                oldVal,
                                null,
                                storageRevision,
                                futures,
                                ConfigurationNamedListListener::onDelete,
                                eventConfigs
                        );

                        // Notification for deleted configuration.

                        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                            notifyPublicListeners(
                                    listeners(any(namedDynamicConfig(anyConfig, key))),
                                    oldVal,
                                    null,
                                    storageRevision,
                                    futures,
                                    ConfigurationListener::onUpdate,
                                    eventConfigs
                            );
                        }

                        notifyPublicListeners(
                                delNodeCfg.listeners(),
                                oldVal,
                                null,
                                storageRevision,
                                futures,
                                ConfigurationListener::onUpdate,
                                eventConfigs
                        );

                        removeConfigContainer(eventConfigs, delNodeCfg);
                    }

                    for (Map.Entry<String, String> entry : renamed.entrySet()) {
                        DynamicConfiguration<InnerNode, ?> renNodeCfg =
                                (DynamicConfiguration<InnerNode, ?>) namedListCfg.members().get(entry.getValue());

                        putConfigContainer(eventConfigs, renNodeCfg, entry.getValue());

                        InnerNode oldVal = oldNamedList.getInnerNode(entry.getKey());
                        InnerNode newVal = newNamedList.getInnerNode(entry.getValue());

                        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                            notifyPublicListeners(
                                    extendedListeners(namedDynamicConfig(anyConfig, key)),
                                    oldVal,
                                    newVal,
                                    storageRevision,
                                    futures,
                                    (listener, evt) -> listener.onRename(entry.getKey(), entry.getValue(), evt),
                                    eventConfigs
                            );
                        }

                        notifyPublicListeners(
                                namedDynamicConfig(cfgNode, key).extendedListeners(),
                                oldVal,
                                newVal,
                                storageRevision,
                                futures,
                                (listener, evt) -> listener.onRename(entry.getKey(), entry.getValue(), evt),
                                eventConfigs
                        );

                        removeConfigContainer(eventConfigs, renNodeCfg);
                    }

                    Set<String> updated = new HashSet<>(newNames);
                    updated.retainAll(oldNames);

                    for (String name : updated) {
                        InnerNode oldVal = oldNamedList.getInnerNode(name);
                        InnerNode newVal = newNamedList.getInnerNode(name);

                        if (oldVal == newVal) {
                            continue;
                        }

                        DynamicConfiguration<InnerNode, ?> updNodeCfg =
                                (DynamicConfiguration<InnerNode, ?>) namedListCfgMembers.get(name);

                        putConfigContainer(eventConfigs, updNodeCfg, name);

                        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                            notifyPublicListeners(
                                    extendedListeners(namedDynamicConfig(anyConfig, key)),
                                    oldVal,
                                    newVal,
                                    storageRevision,
                                    futures,
                                    ConfigurationListener::onUpdate,
                                    eventConfigs
                            );
                        }

                        notifyPublicListeners(
                                namedDynamicConfig(cfgNode, key).extendedListeners(),
                                oldVal,
                                newVal,
                                storageRevision,
                                futures,
                                ConfigurationListener::onUpdate,
                                eventConfigs
                        );

                        if (newAnyConfigs == null) {
                            newAnyConfigs = mergeAnyConfigs(
                                    viewReadOnly(anyConfigs, cfg -> any(namedDynamicConfig(cfg, key))),
                                    any(namedListCfg)
                            );
                        }

                        notifyListeners(
                                oldVal,
                                newVal,
                                updNodeCfg,
                                storageRevision,
                                futures,
                                newAnyConfigs,
                                eventConfigs,
                                false
                        );
                    }
                }

                return null;
            }
        }, true);

        removeConfigContainer(eventConfigs, cfgNode);
    }

    /**
     * Invoke {@link ConfigurationListener#onUpdate(ConfigurationNotificationEvent)} on all passed listeners and put results in {@code
     * futures}. Not recursively.
     *
     * @param listeners       Collection of listeners.
     * @param oldVal          Old configuration value.
     * @param newVal          New configuration value.
     * @param storageRevision Storage revision.
     * @param futures         Write-only list of futures.
     * @param updater         Update closure to be invoked on the listener instance.
     * @param <V>             Type of the node.
     * @param <L>             Type of the configuration listener.
     */
    private static <V, L extends ConfigurationListener<V>> void notifyPublicListeners(
            Collection<? extends L> listeners,
            @Nullable V oldVal,
            V newVal,
            long storageRevision,
            List<CompletableFuture<?>> futures,
            BiFunction<L, ConfigurationNotificationEvent<V>, CompletableFuture<?>> updater,
            Map<Class<?>, ConfigurationContainer> configs
    ) {
        if (listeners.isEmpty()) {
            return;
        }

        ConfigurationNotificationEvent<V> evt = new ConfigurationNotificationEventImpl<>(
                oldVal,
                newVal,
                storageRevision,
                configs
        );

        for (L listener : listeners) {
            try {
                CompletableFuture<?> future = updater.apply(listener, evt);

                assert future != null : updater;

                if (future.isCompletedExceptionally() || future.isCancelled() || !future.isDone()) {
                    futures.add(future);
                }
            } catch (Throwable t) {
                futures.add(CompletableFuture.failedFuture(t));
            }
        }
    }

    /**
     * Ensures that dynamic configuration tree is up to date and further notifications on it will be invoked correctly.
     *
     * @param cfg Dynamic configuration node instance.
     */
    public static void touch(DynamicConfiguration<?, ?> cfg) {
        cfg.touchMembers();

        for (ConfigurationProperty<?> value : cfg.members().values()) {
            if (value instanceof DynamicConfiguration) {
                touch((DynamicConfiguration<?, ?>) value);
            }
        }
    }

    /**
     * Recursive notification of all public listeners from the {@code anyConfigs} for the created naming list item, accumulating resulting
     * futures in {@code futures} list.
     *
     * @param innerNode       Configuration values root.
     * @param cfgNode         Public configuration tree node corresponding to the current inner nodes.
     * @param storageRevision Storage revision.
     * @param futures         Write-only list of futures to accumulate results.
     * @param anyConfigs      Current {@link NamedListConfiguration#any "any"} configurations.
     */
    private static void notifyAnyListenersOnCreate(
            InnerNode innerNode,
            DynamicConfiguration<InnerNode, ?> cfgNode,
            long storageRevision,
            List<CompletableFuture<?>> futures,
            Collection<DynamicConfiguration<InnerNode, ?>> anyConfigs,
            Map<Class<?>, ConfigurationContainer> configs
    ) {
        assert !(cfgNode instanceof NamedListConfiguration);

        for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
            notifyPublicListeners(
                    anyConfig.listeners(),
                    null,
                    innerNode,
                    storageRevision,
                    futures,
                    ConfigurationListener::onUpdate,
                    configs
            );
        }

        innerNode.traverseChildren(new ConfigurationVisitor<Void>() {
            /** {@inheritDoc} */
            @Override
            public Void visitLeafNode(String key, Serializable newLeaf) {
                for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                    notifyPublicListeners(
                            listeners(dynamicProperty(anyConfig, key)),
                            null,
                            newLeaf,
                            storageRevision,
                            futures,
                            ConfigurationListener::onUpdate,
                            configs
                    );
                }

                return null;
            }

            /** {@inheritDoc} */
            @Override
            public Void visitInnerNode(String key, InnerNode newNode) {
                DynamicConfiguration<InnerNode, ?> innerNodeCfg = dynamicConfig(cfgNode, key);

                configs.put(innerNodeCfg.configType(), new ConfigurationContainer(null, innerNodeCfg));

                notifyAnyListenersOnCreate(
                        newNode,
                        innerNodeCfg,
                        storageRevision,
                        futures,
                        viewReadOnly(anyConfigs, cfg -> dynamicConfig(cfg, key)),
                        configs
                );

                configs.remove(innerNodeCfg.configType());

                return null;
            }

            /** {@inheritDoc} */
            @Override
            public Void visitNamedListNode(String key, NamedListNode<?> newNamedList) {
                for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                    notifyPublicListeners(
                            listeners(namedDynamicConfig(anyConfig, key)),
                            null,
                            (NamedListView<InnerNode>) newNamedList,
                            storageRevision,
                            futures,
                            ConfigurationListener::onUpdate,
                            configs
                    );
                }

                // Created only.

                // Lazy initialization.
                Collection<DynamicConfiguration<InnerNode, ?>> newAnyConfigs = null;

                for (String name : newNamedList.namedListKeys()) {
                    DynamicConfiguration<InnerNode, ?> newNodeCfg =
                            (DynamicConfiguration<InnerNode, ?>) namedDynamicConfig(cfgNode, key).getConfig(name);

                    configs.put(newNodeCfg.configType(), new ConfigurationContainer(name, newNodeCfg));

                    InnerNode newVal = newNamedList.getInnerNode(name);

                    for (DynamicConfiguration<InnerNode, ?> anyConfig : anyConfigs) {
                        notifyPublicListeners(
                                extendedListeners(namedDynamicConfig(anyConfig, key)),
                                null,
                                newVal,
                                storageRevision,
                                futures,
                                ConfigurationNamedListListener::onCreate,
                                configs
                        );
                    }

                    if (newAnyConfigs == null) {
                        newAnyConfigs = mergeAnyConfigs(
                                viewReadOnly(anyConfigs, cfg -> any(namedDynamicConfig(cfg, key))),
                                any(namedDynamicConfig(cfgNode, key))
                        );
                    }

                    notifyAnyListenersOnCreate(
                            newVal,
                            newNodeCfg,
                            storageRevision,
                            futures,
                            newAnyConfigs,
                            configs
                    );

                    configs.remove(newNodeCfg.configType());
                }

                return null;
            }
        }, true);
    }

    /**
     * Merge {@link NamedListConfiguration#any "any"} configurations.
     *
     * @param anyConfigs Current {@link NamedListConfiguration#any "any"} configurations.
     * @param anyConfig  New {@link NamedListConfiguration#any "any"} configuration.
     * @return Merged {@link NamedListConfiguration#any "any"} configurations.
     */
    private static Collection<DynamicConfiguration<InnerNode, ?>> mergeAnyConfigs(
            Collection<DynamicConfiguration<InnerNode, ?>> anyConfigs,
            @Nullable DynamicConfiguration<InnerNode, ?> anyConfig
    ) {
        return Stream.concat(anyConfigs.stream(), Stream.of(anyConfig))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * Returns the dynamic property of the leaf.
     *
     * @param dynamicConfig Dynamic configuration.
     * @param nodeName      Name of the child node.
     * @return Dynamic property of a leaf or {@code null} if the leaf does not exist.
     */
    private static @Nullable DynamicProperty<Serializable> dynamicProperty(
            DynamicConfiguration<InnerNode, ?> dynamicConfig,
            String nodeName
    ) {
        return (DynamicProperty<Serializable>) dynamicConfig.members().get(nodeName);
    }

    /**
     * Returns the dynamic configuration of the child node.
     *
     * @param dynamicConfig Dynamic configuration.
     * @param nodeName      Name of the child node.
     * @return Dynamic configuration of the child node or {@code null} if the child node does not exist.
     */
    private static @Nullable DynamicConfiguration<InnerNode, ?> dynamicConfig(
            DynamicConfiguration<InnerNode, ?> dynamicConfig,
            String nodeName
    ) {
        return (DynamicConfiguration<InnerNode, ?>) dynamicConfig.members().get(nodeName);
    }

    /**
     * Returns the named dynamic configuration of the child node.
     *
     * @param dynamicConfig Dynamic configuration.
     * @param nodeName      Name of the child node.
     * @return Named dynamic configuration of the child node or {@code null} if the child node does not exist.
     */
    private static @Nullable NamedListConfiguration<?, InnerNode, ?> namedDynamicConfig(
            DynamicConfiguration<InnerNode, ?> dynamicConfig,
            String nodeName
    ) {
        return (NamedListConfiguration<?, InnerNode, ?>) dynamicConfig.members().get(nodeName);
    }

    /**
     * Returns the dynamic configuration of the {@link NamedListConfiguration#any any} node.
     *
     * @param namedConfig Dynamic configuration.
     * @return Dynamic configuration of the "any" node.
     */
    private static @Nullable DynamicConfiguration<InnerNode, ?> any(@Nullable NamedListConfiguration<?, InnerNode, ?> namedConfig) {
        return namedConfig == null ? null : (DynamicConfiguration<InnerNode, ?>) namedConfig.any();
    }

    /**
     * Null-safe version of {@link ConfigurationNode#listeners()}. Needed for working with "any" configuration properties, see this class'
     * javadoc for details.
     *
     * @return Listeners of the given node or an empty list if it is {@code null}.
     */
    private static <T> Collection<ConfigurationListener<T>> listeners(@Nullable ConfigurationNode<T> node) {
        return node == null ? List.of() : node.listeners();
    }

    /**
     * Null-safe version of {@link NamedListConfiguration#extendedListeners()}. Needed for working with "any" configuration properties, see
     * this class' javadoc for details.
     *
     * @return Listeners of the given node or an empty list if it is {@code null}.
     */
    private static <T> Collection<ConfigurationNamedListListener<T>> extendedListeners(@Nullable NamedListConfiguration<?, T, ?> node) {
        return node == null ? List.of() : node.extendedListeners();
    }

    /**
     * Adds a configuration container for {@link ConfigurationNotificationEvent}.
     *
     * <p>NOTE: Addition for {@link DynamicConfiguration#configType}, {@link DynamicConfiguration#internalConfigTypes}
     * and {@link DynamicConfiguration#polymorphicInstanceConfigType}.
     *
     * @param containers Configuration containers. Mapping: Configuration interface -> container.
     * @param cfg Configuration.
     * @param keyNamedCfg Key of the named list item for {@link ConfigurationNotificationEvent#name}.
     */
    private static void putConfigContainer(
            Map<Class<?>, ConfigurationContainer> containers,
            DynamicConfiguration<InnerNode, ?> cfg,
            @Nullable String keyNamedCfg
    ) {
        ConfigurationContainer container = new ConfigurationContainer(keyNamedCfg, cfg.isRemovedFromNamedList() ? null : cfg);

        containers.put(cfg.configType(), container);

        Class<?>[] internalConfigTypes = cfg.internalConfigTypes();

        if (internalConfigTypes != null) {
            for (int i = 0; i < internalConfigTypes.length; i++) {
                containers.put(internalConfigTypes[i], container);
            }
        }

        Class<?> polymorphicInstanceConfigType = cfg.polymorphicInstanceConfigType();

        if (polymorphicInstanceConfigType != null) {
            containers.put(
                    polymorphicInstanceConfigType,
                    new ConfigurationContainer(keyNamedCfg, cfg.isRemovedFromNamedList() ? null : cfg.specificConfigTree())
            );
        }
    }

    /**
     * Removes a configuration container for {@link ConfigurationNotificationEvent}.
     *
     * <p>NOTE: Removal for {@link DynamicConfiguration#configType}, {@link DynamicConfiguration#internalConfigTypes}
     * and {@link DynamicConfiguration#polymorphicInstanceConfigType}.
     *
     * @param containers Configuration containers. Mapping: Configuration interface -> container.
     * @param cfg Configuration.
     */
    private static void removeConfigContainer(
            Map<Class<?>, ConfigurationContainer> containers,
            DynamicConfiguration<InnerNode, ?> cfg
    ) {
        containers.remove(cfg.configType());

        Class<?>[] internalConfigTypes = cfg.internalConfigTypes();

        if (internalConfigTypes != null) {
            for (int i = 0; i < internalConfigTypes.length; i++) {
                containers.remove(internalConfigTypes[i]);
            }
        }

        Class<?> polymorphicInstanceConfigType = cfg.polymorphicInstanceConfigType();

        if (polymorphicInstanceConfigType != null) {
            containers.remove(polymorphicInstanceConfigType);
        }
    }
}
