/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Data.Common;
using System.Text;
using System.ComponentModel;

using PostgreSql.Data.NPgClient;

namespace PostgreSql.Data.PgSqlClient
{
	public sealed class PgCommandBuilder : Component
	{
		#region Fields

		private PgDataAdapter	dataAdapter;
		private string			sqlInsert;
		private string			sqlUpdate;
		private string			sqlDelete;
		private PgCommand		insertCommand;
		private PgCommand		updateCommand;
		private PgCommand		deleteCommand;		
		private	string			separator;
		// private string			whereClausule1;
		private string			whereClausule2;
		private string			setClausule;
		private DataTable		schemaTable;
		private string			tableName;
		private bool			hasPrimaryKey;
		private bool			hasUniqueKey;
		private string			quotePrefix;
		private string			quoteSuffix;
		private bool			disposed;

		#endregion

		#region Properties

		[DefaultValue(null)]
		public PgDataAdapter DataAdapter
		{
			get { return dataAdapter; }
			set
			{			
				dataAdapter	= value;

				// Registers the CommandBuilder as a listener for RowUpdating events that are 
				// generated by the PgDataAdapter specified in this property.
				if (dataAdapter != null)
				{
					dataAdapter.RowUpdating += new PgRowUpdatingEventHandler(rowUpdatingHandler);
				}
			}
		}
		
		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public string QuotePrefix
		{
			get { return quotePrefix; }
			set
			{
				if (insertCommand != null || updateCommand != null || deleteCommand != null)
				{
					throw new InvalidOperationException("This property cannot be changed after an insert, update, or delete command has been generated.");
				}
				
				quotePrefix = value;
			}
		}

		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public string QuoteSuffix
		{
			get { return quoteSuffix; }
			set
			{
				if (insertCommand != null || updateCommand != null || deleteCommand != null)
				{
					throw new InvalidOperationException("This property cannot be changed after an insert, update, or delete command has been generated.");
				}
				
				quoteSuffix = value;
			}
		}

		private PgCommand selectCommand
		{
			get
			{
				if (this.dataAdapter.SelectCommand != null)
				{
					return this.dataAdapter.SelectCommand;
				}

				return null;
			}
		}

		#endregion

		#region Constructors

		public PgCommandBuilder() : base()
		{			
			sqlInsert		= "INSERT INTO {0} ({1}) VALUES ({2})";
			sqlUpdate		= "UPDATE {0} SET {1} WHERE ( {2} )";
			sqlDelete		= "DELETE FROM {0} WHERE ( {1} )";
			// whereClausule1	= "(({0} IS NULL AND ${1} = NULL) OR ({0} = ${2}))";			
			whereClausule2	= "({0} = ${1})";
			setClausule		= "{0} = ${1}";
			separator		= ", ";
			quotePrefix		= String.Empty;
			quoteSuffix		= String.Empty;

			GC.SuppressFinalize(this);
		}
		
		public PgCommandBuilder(PgDataAdapter adapter) : this()
		{
			this.DataAdapter    = adapter;
		}

		#endregion

		#region IDisposable Methods

		protected override void Dispose(bool disposing)
		{
			if (!disposed)
			{
				try
				{
					if (disposing)
					{
						if (insertCommand != null)
						{
							insertCommand.Dispose();
						}
						if (updateCommand != null)
						{
							updateCommand.Dispose();
						}
						if (deleteCommand != null)
						{
							deleteCommand.Dispose();
						}
						if (schemaTable != null)
						{
							schemaTable.Dispose();
						}
					}
					
					// release any unmanaged resources
					
					disposed = true;
				}
				finally 
				{
					base.Dispose(disposing);
				}
			}
		}

		#endregion

		#region Static Methods

		public static void DeriveParameters(PgCommand command)
		{
			if (command.CommandType != CommandType.StoredProcedure)
			{
				throw new InvalidOperationException("The command text is not a valid stored procedure name.");
			}

			command.Parameters.Clear();

			DataTable spSchema = command.Connection.GetDbSchemaTable(PgDbSchemaType.Functions,
				new object[] {null, command.CommandText.ToLower()});
			
			if (spSchema.Rows.Count != 0)
			{
				int[] parameterTypes = (int[])spSchema.Rows[0]["ARGUMENTS"];
				int	  parameterCount = (short)spSchema.Rows[0]["ARGUMENT_NUMBER"];

				for (int i = 0; i < parameterCount; i++)
				{
					PgParameter parameter = command.Parameters.Add(
						"@ip" + i.ToString(),
						(PgDbType)PgDbClient.Types[parameterTypes[i]].DataType);

					parameter.Direction = ParameterDirection.Input;
				}
				
				int returnType = (int)spSchema.Rows[0]["RETURN_TYPE"];
				if (returnType != 0)
				{
					PgParameter parameter = command.Parameters.Add(
						"@op0",
						PgDbClient.Types[returnType]);

					parameter.Direction = ParameterDirection.Output;
				}
			}
		}

		#endregion

		#region Methods
		
		public PgCommand GetInsertCommand()
		{			
			if (this.insertCommand == null)
			{
				bool mustClose = false;
				try
				{
					if (this.selectCommand.Connection.State == ConnectionState.Closed)
					{
						mustClose = true;
						this.selectCommand.Connection.Open();						
					}

					this.buildInsertCommand(null, null);
				}
				catch (Exception ex)
				{
					throw ex;
				}
				finally
				{
					if (mustClose)
					{
						this.selectCommand.Connection.Close();
					}
				}
			}
			
			return insertCommand;
		}

		public PgCommand GetUpdateCommand()
		{			
			if (this.updateCommand == null)
			{
				bool mustClose = false;
				try
				{
					if (this.selectCommand.Connection.State == ConnectionState.Closed)
					{
						mustClose = true;
						this.selectCommand.Connection.Open();						
					}

					this.buildUpdateCommand(null, null);
				}
				catch (Exception ex)
				{
					throw ex;
				}
				finally
				{
					if (mustClose)
					{
						this.selectCommand.Connection.Close();
					}
				}
			}
			
			return updateCommand;
		}

		public PgCommand GetDeleteCommand()
		{			
			if (deleteCommand == null)
			{
				bool mustClose = false;
				try
				{
					if (this.selectCommand.Connection.State == ConnectionState.Closed)
					{
						mustClose = true;
						this.selectCommand.Connection.Open();						
					}

					this.buildDeleteCommand(null, null);
				}
				catch (Exception ex)
				{
					throw ex;
				}
				finally
				{
					if (mustClose)
					{
						this.selectCommand.Connection.Close();
					}
				}
			}
			
			return deleteCommand;
		}
		
		public void RefreshSchema()
		{
			schemaTable   = null;
			insertCommand = null;
			updateCommand = null;
			deleteCommand = null;
		}

		#endregion

		#region Build Command Methods

		private PgCommand buildInsertCommand(DataRow row, DataTableMapping tableMapping)
		{						
			StringBuilder	sql		= new StringBuilder();
			StringBuilder	fields	= new StringBuilder();
			StringBuilder	values	= new StringBuilder();
			string			dsColumnName = String.Empty;

			this.buildSchemaTable();

			insertCommand = new PgCommand(sql.ToString(), selectCommand.Connection, selectCommand.Transaction);

			int i = 1;
			foreach (DataRow schemaRow in schemaTable.Rows)
			{				
				if (isUpdatable(schemaRow, row))
				{
					if (fields.Length > 0)
					{
						fields.Append(separator);
					}
					if (values.Length > 0)
					{
						values.Append(separator);
					}
					
					PgParameter parameter = createParameter(schemaRow, i, false);

					if (row != null)
					{
						dsColumnName	= getColumnName(tableMapping, schemaRow["BaseColumnName"].ToString());
						parameter.Value = row[dsColumnName];
					}

					i++;

					// Build Field name and append it to the string					
					fields.Append(getQuotedIdentifier(schemaRow["BaseColumnName"]));
					
					// Build value name and append it to the string
					values.Append(parameter.ParameterName);

					insertCommand.Parameters.Add(parameter);
				}
			}

			sql.AppendFormat(sqlInsert, getQuotedIdentifier(tableName), fields.ToString(), values.ToString());

			insertCommand.CommandText = sql.ToString();

			// None is the Default value for automatically generated commands
			insertCommand.UpdatedRowSource = UpdateRowSource.None;

			return insertCommand;
		}
		
		private PgCommand buildUpdateCommand(DataRow row, DataTableMapping tableMapping)
		{
			StringBuilder sql			= new StringBuilder();
			StringBuilder sets			= new StringBuilder();
			StringBuilder where			= new StringBuilder();
			string		  dsColumnName	= String.Empty;

			buildSchemaTable();

			if (!hasPrimaryKey && !hasUniqueKey)
			{
				throw new InvalidOperationException("Dynamic SQL generation for the UpdateCommand is not supported against a SelectCommand that does not return any key column information.");
			}

			updateCommand = new PgCommand(sql.ToString(), selectCommand.Connection, selectCommand.Transaction);

			int i = 1;
			foreach (DataRow schemaRow in schemaTable.Rows)
			{				
				if (isUpdatable(schemaRow, row))
				{
					if (sets.Length > 0)
					{
						sets.Append(separator);
					}

					// Build Field name and append it to the string
					sets.AppendFormat(setClausule, 
						getQuotedIdentifier(schemaRow["BaseColumnName"]), i);

					PgParameter parameter = createParameter(schemaRow, i, false);
										
					if (row != null)
					{
						dsColumnName	= this.getColumnName(tableMapping, schemaRow["BaseColumnName"].ToString());
						parameter.Value = row[dsColumnName];
					}

					i++;

					updateCommand.Parameters.Add(parameter);
				}
			}
			
			// Build where clausule
			foreach (DataRow schemaRow in schemaTable.Rows)
			{				
				if (includedInWhereClause(schemaRow)) 
				{
					if (where.Length > 0)
					{
						where.Append(" AND ");
					}
					
					dsColumnName = this.getColumnName(tableMapping, schemaRow["BaseColumnName"].ToString());

					string quotedId = getQuotedIdentifier(schemaRow["BaseColumnName"]);

					PgParameter parameter = createParameter(schemaRow, i, true);
					where.AppendFormat(whereClausule2, quotedId, i);

					if (row != null)
					{
						parameter.Value = row[dsColumnName, DataRowVersion.Original];
					}

					updateCommand.Parameters.Add(parameter);

					i++;
				}
			}

			sql.AppendFormat(sqlUpdate, getQuotedIdentifier(tableName), sets.ToString(), where.ToString());
			
			updateCommand.CommandText = sql.ToString();

			// None is the Default value for automatically generated commands
			updateCommand.UpdatedRowSource = UpdateRowSource.None;

			return updateCommand;
		}

		private PgCommand buildDeleteCommand(DataRow row, DataTableMapping tableMapping)
		{
			StringBuilder sql	= new StringBuilder();
			StringBuilder where = new StringBuilder();
			string		  dsColumnName = String.Empty;

			buildSchemaTable();

			if (!hasPrimaryKey && !hasUniqueKey)
			{
				throw new InvalidOperationException("Dynamic SQL generation for the DeleteCommand is not supported against a SelectCommand that does not return any key column information.");
			}

			deleteCommand = new PgCommand(sql.ToString(), selectCommand.Connection, selectCommand.Transaction);
		
			// Build where clausule
			int i = 1;
			foreach (DataRow schemaRow in schemaTable.Rows)
			{				
				if (includedInWhereClause(schemaRow)) 
				{
					if (where.Length > 0)
					{
						where.Append(" AND ");
					}
					
					dsColumnName = this.getColumnName(tableMapping, schemaRow["BaseColumnName"].ToString());

					string quotedId = getQuotedIdentifier(schemaRow["BaseColumnName"]);

					PgParameter parameter = createParameter(schemaRow, i, true);
					where.AppendFormat(whereClausule2, quotedId, i);

					if (row != null)
					{
						parameter.Value = row[dsColumnName, DataRowVersion.Original];
					}

					deleteCommand.Parameters.Add(parameter);

					i++;
				}
			}

			sql.AppendFormat(sqlDelete, getQuotedIdentifier(tableName), where.ToString());
			
			deleteCommand.CommandText = sql.ToString();

			// None is the Default value for automatically generated commands
			deleteCommand.UpdatedRowSource = UpdateRowSource.None;

			return deleteCommand;
		}

		private string getColumnName(DataTableMapping tableMapping, string sourceColumn)
		{
			string dsColumnName;

			if (tableMapping != null)
			{
				if (tableMapping.ColumnMappings.Count > 0)
				{
					dsColumnName = tableMapping.ColumnMappings[sourceColumn].DataSetColumn;
				}
				else
				{
					dsColumnName = sourceColumn;
				}
			}
			else
			{
				dsColumnName = sourceColumn;
			}

			return dsColumnName;
		}

		private PgParameter createParameter(DataRow schemaRow, int index, bool isWhereParameter)
		{
			PgParameter parameter = new PgParameter(String.Format("@p{0}", index), (PgDbType)schemaRow["ProviderType"]);

			parameter.Size = Convert.ToInt32(schemaRow["ColumnSize"]);
			if (schemaRow["NumericPrecision"] != DBNull.Value)
			{
				parameter.Precision	= Convert.ToByte(schemaRow["NumericPrecision"]);
			}
			if (schemaRow["NumericScale"] != DBNull.Value)
			{
				parameter.Precision	= Convert.ToByte(schemaRow["NumericScale"]);
			}
			parameter.SourceColumn	= Convert.ToString(schemaRow["BaseColumnName"]);

			if (isWhereParameter)
			{
				parameter.SourceVersion	= DataRowVersion.Original;
			}
			else
			{
				parameter.SourceVersion	= DataRowVersion.Current;
			}

			return parameter;
		}

		private bool isUpdatable(DataRow schemaRow, DataRow row)
		{
			if (row != null)
			{
				string		columnName	= (string) schemaRow["ColumnName"];
				DataColumn	column		= row.Table.Columns[columnName];

				if (column != null)
				{
					if (column.Expression != null && column.Expression.Length != 0)
					{
						return false;
					}
					if (column.AutoIncrement)
					{
						return false;
					}
					if (column.ReadOnly)
					{
						return false;
					}
				}
			}

			if ((bool) schemaRow["IsExpression"])
			{
				return false;
			}
			if ((bool) schemaRow["IsAutoIncrement"])
			{
				return false;
			}
			if ((bool) schemaRow["IsRowVersion"])
			{
				return false;
			}
			if ((bool) schemaRow["IsReadOnly"])
			{
				return false;
			}

			return true;
		}

		private bool includedInWhereClause(DataRow schemaRow)
		{
			/*
			PgDbType pgDbType= (PgDbType)schemaRow["ProviderType"];

			if (pgDbType == PgDbType.Binary)
			{
				return false;
			}

			if (pgDbType == PgDbType.Array)
			{
				return false;
			}
			*/

			if (!(bool)schemaRow["IsKey"])
			{
				return false;
			}

			return true;
		}

		private void buildSchemaTable()
		{
			if (selectCommand == null)
			{
				throw new InvalidOperationException("The DataAdapter.SelectCommand property needs to be initialized.");
			}
			if (selectCommand.Connection == null)
			{
				throw new InvalidOperationException("The DataAdapter.selectCommand.Connection property needs to be initialized.");
			}

			if (schemaTable == null)
			{				
				PgDataReader reader = selectCommand.ExecuteReader(CommandBehavior.SchemaOnly);
				schemaTable = reader.GetSchemaTable();
				reader.Close();

				checkSchemaTable();
			}			
		}

		private void checkSchemaTable()
		{
			tableName		= String.Empty;
			hasPrimaryKey	= false;
			hasUniqueKey	= false;

			foreach (DataRow schemaRow in schemaTable.Rows)
			{
				if (tableName.Length == 0)
				{
					tableName = (string)schemaRow["BaseSchemaName"] + "." +
								(string)schemaRow["BaseTableName"];
				}
				if (!(bool)schemaRow["IsExpression"] && tableName != (string)schemaRow["BaseSchemaName"] + "." + (string)schemaRow["BaseTableName"])
				{
					throw new InvalidOperationException("Dynamic SQL generation is not supported against multiple base tables.");
				}
				if ((bool)schemaRow["IsKey"])
				{
					hasPrimaryKey = true;
				}
				if ((bool)schemaRow["IsUnique"])
				{
					hasUniqueKey = true;
				}
			}
		}

		private string getQuotedIdentifier(object identifier)
		{
			return quotePrefix + identifier.ToString() + quoteSuffix;
		}

		#endregion

		#region Event Handler Methods

		private void rowUpdatingHandler(object sender, PgRowUpdatingEventArgs e)
		{
			if (e.Status != UpdateStatus.Continue)
			{
				return;
			}

			switch (e.StatementType) 
			{
				case StatementType.Insert:
					insertCommand = e.Command;
					break;

				case StatementType.Update:
					updateCommand = e.Command;
					break;

				case StatementType.Delete:
					deleteCommand = e.Command;
					break;

				default:
					return;
			}

			bool mustClose = false;
			try
			{
				switch (e.StatementType) 
				{
					case StatementType.Insert:
						e.Command = this.buildInsertCommand(e.Row, e.TableMapping);
						break;

					case StatementType.Update:
						e.Command = this.buildUpdateCommand(e.Row, e.TableMapping);
						break;
					
					case StatementType.Delete:
						e.Command = this.buildDeleteCommand(e.Row, e.TableMapping);
						break;
				}
			}
			catch (Exception exception) 
			{
				e.Errors = exception;
				e.Status = UpdateStatus.ErrorsOccurred;
			}
			finally
			{
				if (mustClose)
				{
					this.dataAdapter.SelectCommand.Connection.Close();
				}
			}
		}

		#endregion
	}
}
