/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Text;

namespace PostgreSql.Data.PgSqlClient.DbSchema
{
	internal class PgCheckConstraintsByTableSchema : PgAbstractDbSchema
	{
		#region Constructors

		public PgCheckConstraintsByTableSchema() : base("CheckConstraintsByTable")
		{
		}

		#endregion

		#region Add Methods

		public override void AddTables()
		{
			AddTable("pg_constraint");
		}

		public override void AddRestrictionColumns()
		{
			AddRestrictionColumn("pg_namespace.nspname"	, "CONSTRAINT_SCHEMA"	, null);
			AddRestrictionColumn("pg_constraint.conname", "CONSTRAINT_NAME"		, null);
			AddRestrictionColumn("tbn.nspname"			, "TABLE_SCHEMA"		, null);
			AddRestrictionColumn("pg_class.relname"		, "TABLE_NAME"			, null);
		}

		public override void AddDataColumns()
		{
			AddDataColumn("pg_get_constraintdef(pg_constraint.oid)", "CHECK_CLAUSULE");
			AddDataColumn("pg_description.description", "DESCRIPTION");						
		}

		public override void AddJoins()
		{
			AddJoin("left join", "pg_class"			, "pg_class.oid = pg_constraint.conrelid");
			AddJoin("left join", "pg_namespace tbn"	, "pg_class.relnamespace = tbn.oid");
			AddJoin("left join", "pg_namespace"		, "pg_constraint.connamespace = pg_namespace.oid");
			AddJoin("left join", "pg_description"	, "pg_constraint.oid = pg_description.objoid");
		}

		public override void AddOrderByColumns()
		{
			AddOrderBy("pg_namespace.nspname");
			AddOrderBy("pg_class.relname");
			AddOrderBy("pg_constraint.conname");			
		}

		public override void AddWhereFilters()
		{
			AddWhereFilter("pg_constraint.contype = 'c'");
			AddWhereFilter("pg_class.relkind = 'r'");
		}

		#endregion

		#region Parse Methods

		public override object[] ParseRestrictions(object[] restrictions)
		{
			object[] parsed = restrictions;

			if (parsed != null)
			{
				if (parsed.Length == 7 && parsed[6] != null)
				{
					switch (parsed[6].ToString().ToUpper())
					{
						case "UNIQUE":
							parsed[3] = "u";
							break;

						case "PRIMARY KEY":
							parsed[3] = "p";
							break;

						case "FOREIGN KEY":
							parsed[3] = "f";
							break;

						case "CHECK":
							parsed[3] = "c";
							break;
					}
				}
			}

			return parsed;
		}

		#endregion

		#region Private Methods

		private string getConstraintTypeExpression(string fieldName)
		{
			StringBuilder expression = new StringBuilder();

			expression.AppendFormat(" case {0} ", fieldName);
			expression.Append(" when 'u' THEN 'UNIQUE'");
			expression.Append(" when 'p' THEN 'PRIMARY KEY'");
			expression.Append(" when 'f' THEN 'FOREIGN KEY'");
			expression.Append(" when 'c' THEN 'CHECK'");
			expression.Append(" END ");

			return expression.ToString();
		}

		#endregion
	}
}