package org.springframework.uaa.client.internal;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.Security;
import java.util.prefs.Preferences;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPUtil;
import org.springframework.uaa.client.util.Assert;
import org.springframework.uaa.client.util.PreferencesUtils;
import org.springframework.uaa.client.util.StreamUtils;
import org.springframework.uaa.client.util.XmlUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Utility class that handles updates to internal UAA configuration parameters. This implementation expects two
 * {@link InputStream}s to {@link #updateConfiguration(InputStream, InputStream)} of which the first streams
 * represents the contents of <code>uaa-client.xml</code> and the 2nd the PGP signature of the configuration file.
 * 
 * <p>
 * Only if the PGP signature can be successfully validated using the <code>uaa.asc</code> public key, this
 * implementation will consider the configuration content trusted and proceed with updating the parameters that are
 * found in the following XML structure:
 * 
 * <pre>
 * <uaa version="3">
 * 	<configuration>
 * 		<!-- Configuration for TransmissionAwareUaaServiceImpl -->
 * 		<node path="org/springframework/uaa/client/internal">
 * 			<!-- 24hrs: 1000L * 60L * 60L * 24L -->
 * 			<key name="download_interval" value="86400000" />
 * 			<!-- 24hrs: 1000L * 60L * 60L * 24L -->
 * 			<key name="ping_interval" value="86400000" />
 * 			<!-- 1hr: 1000L * 60L * 60L -->
 * 			<key name="upload_interval" value="3600000" />		
 * 		</node>
 * 	</configuration>
 * </uaa>
 * </pre>
 * 
 * <p>
 * For each <code>node</code> element the corresponding user node in the {@link Preferences} API will be loaded.
 * The {@link Preferences} node is identified by the contents of the <code>path</code> attribute.
 * 
 * @author Christian Dupuis
 * @since 1.0.1
 */
public abstract class UaaConfigurationProcessor {

	static final URL SIGNATURE_URL;
	static final URL UAA_URL;
	static final String CONFIGURATION_VERSION_KEY = "configuration_version";

	private static final String KEY_NAME = "/org/springframework/uaa/client/keys/uaa.asc";
	private static final Preferences P = PreferencesUtils.getPreferencesFor(UaaConfigurationProcessor.class);

	static {
		// Register the BC provider
		Security.addProvider(new BouncyCastleProvider());

		// Produce the urls safely
		try {
			UAA_URL = new URL("http://uaa.springsource.org/uaa-client.xml");
			SIGNATURE_URL = new URL("http://uaa.springsource.org/uaa-client.xml.asc");
		}
		catch (MalformedURLException neverHappens) {
			throw new IllegalStateException(neverHappens);
		}
	}

	/**
	 * Obtains the {@link PGPPublicKey} from the given {@link InputStream}. 
	 */
	private static PGPPublicKeyRing getPublicKey(InputStream in) {
		Object obj;
		try {
			PGPObjectFactory pgpFact = new PGPObjectFactory(PGPUtil.getDecoderStream(in));
			obj = pgpFact.nextObject();
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}

		if (obj != null && obj instanceof PGPPublicKeyRing) {
			PGPPublicKeyRing keyRing = (PGPPublicKeyRing) obj;
			return keyRing;
		}

		throw new IllegalStateException("Public key not available");
	}
	
	/**
	 * Reads and stores the configuration parameter found in the given {@link Element} into the {@link Preferences} API.  
	 */
	private static void storeConfigurationFromNode(Element elem) {
		String path = elem.getAttribute("path");
		// Obtain Preferences node by the given path
		Preferences p = Preferences.userRoot().node(path);
		
		NodeList keys = elem.getElementsByTagName("key");
		for (int i = 0; i < keys.getLength(); i++) {
			if (keys.item(i) instanceof Element) {
				Element keyElem = (Element) keys.item(i);
				String key = keyElem.getAttribute("name");
				String value = keyElem.getAttribute("value");
				if (key.length() > 0 && value.length() > 0) {
					p.put(key, value);
				}
			}
		}
	}

	/**
	 * Reads all the configuration nodes from the given {@link NodeList} and calls {@link #storeConfigurationFromNode(Element)}
	 * for each contained node. 
	 */
	private static void storeConfigurationFromNodes(NodeList nodes) {
		for (int i = 0; i < nodes.getLength(); i++) {
			Node node = nodes.item(i);
			if (node instanceof Element) {
				storeConfigurationFromNode((Element) node);
			}
		}
	}

	/**
	 * Reads the configuration given by <code>configuration</code>. This method will first check the signature
	 * presented in <code>configurtionSignature</code> if it verfies against the configuration stream.
	 * 
	 * <p>
	 * If the signature cannot be verified successfully the configuration is not updated.
	 * 
	 * @return <code>true</code> if the configuration was successfully updated
	 */
	static final boolean updateConfiguration(InputStream configuration, InputStream configurationSignature) {
		Assert.notNull(configuration, "Configuration required");
		Assert.notNull(configurationSignature, "Configuration signature required");

		InputStream bais = null;
		try {
			bais = StreamUtils.copyIntoResettableStream(configuration);

			// UAA only accepts remote configuration updates if the configuration document is signed
			// with the UAA private key
			if (validateConfigurationSignature(bais, configurationSignature)) {
				bais.reset();
				Document configurationDocument = XmlUtils.parse(bais);
				Element documentElement = configurationDocument.getDocumentElement();

				// Check that locally stored configuration isn't newer as the one that comes in
				// Only if it is newer we want to override the local configuration
				Integer incomingVersion = new Integer(documentElement.getAttribute("version"));
				Integer existingVersion = P.getInt(CONFIGURATION_VERSION_KEY, 0);
				if (incomingVersion > existingVersion) {

					// Process all configuration nodes
					storeConfigurationFromNodes(documentElement.getElementsByTagName("node"));

					// Store new configuration version for later reference
					P.putInt(CONFIGURATION_VERSION_KEY, incomingVersion);
				}

				// Finally we also want to update the detected products
				bais.reset();
				UaaDetectedProductsImpl.setProducts(bais);
				
				return true;
			}
		}
		catch (Exception e) {}
		finally {
			if (bais != null) {
				try { bais.close(); }
				catch (IOException e) {}
			}
		}
		return false;
	}
	
	/**
	 * Validates the presented signature against the contents of <code>resource</code>. This will use
	 * the UAA public key that comes with Spring UAA client.
	 */
	static final boolean validateConfigurationSignature(InputStream resource, InputStream signature) {
		try {
			// Load public key from UAA jar
			PGPPublicKey publicKey = getPublicKey(UaaConfigurationProcessor.class.getResourceAsStream(KEY_NAME))
					.getPublicKey();

			// Load signature and prepare to verify
			PGPObjectFactory factory = new PGPObjectFactory(PGPUtil.getDecoderStream(signature));
			PGPSignatureList p3 = (PGPSignatureList) factory.nextObject();
			PGPSignature pgpSignature = p3.get(0);
			pgpSignature.initVerify(publicKey, "BC");

			// Now verify the signed content
			byte[] buff = new byte[1024];
			int chunk;
			do {
				chunk = resource.read(buff);
				if (chunk > 0) {
					pgpSignature.update(buff, 0, chunk);
				}
			} while (chunk >= 0);

			return pgpSignature.verify();
		}
		catch (Exception e) {
			return false;
		}
	}
	
}
