/*==============================================================================

Project:   bdump
Module:    bdump
Filename:  bdump.c
Author:    Phil Braham (phil@braham.net)
Date:      Dec 2004

Web site:  www.braham.net/Phil/linux_utils.html

Description:
        Dumps a file in hex and ascii for to the terminal.
==============================================================================*/
/*==============================================================================
Include Files
==============================================================================*/
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

/*==============================================================================
Local definitions
==============================================================================*/
typedef int bool;
#define true 1
#define false 0

#define LONGS_PER_LINE_ORD       8
#define LONGS_PER_LINE_WIDE     16
#define LONGS_PER_LINE_NARROW    4

#define FLAGS       "abc::dDfhlmno:rRsu::vwxzZ01"
#define HELP_FLAGS	"abcdDfhlmnorRsvwxzZ01u"

#define MAX_PS_SIZE  12
#define PS_LONG      "8X "
#define PS_SHORT     "4hX "
#define PS_BYTE      "2hX "
#define PS_NZ        "%"
#define PS_ZE        "%0"

#define FI_LONG     "         "
#define FI_SHORT    "     "
#define FI_BYTE     "   "

#define INT_MAX_OFFS 0X7FFFF000	/* Allows for width without overfolwong */

#define VERSION "bdump -  V3.5 22-Dec-2004. Written by P.L.Braham"

#define ESC_CHAR "\033"
#define D_EMP ESC_CHAR "[01;30m"
#define D_RED ESC_CHAR "[01;31m"
#define D_GRE ESC_CHAR "[01;32m"
#define D_YEL ESC_CHAR "[01;33m"
#define D_BLU ESC_CHAR "[01;34m"
#define D_MAG ESC_CHAR "[01;35m"
#define D_CYA ESC_CHAR "[01;36m"
#define D_WHI ESC_CHAR "[01;37m"
#define D_RESET ESC_CHAR "[00m"

const int MaxColours = 7;

/*
    Structure with colour code letter and escape sequence
    to send to terminal.
*/
struct S_Colours
{
    char   charOption;
    char   *escSeq;
} Colours[] =
    {
    'e', D_EMP,
    'r', D_RED,
    'g', D_GRE,
    'y', D_YEL,
    'b', D_BLU,
    'm', D_MAG,
    'c', D_CYA,
    'w', D_WHI
    };

typedef enum
{
    CS_Disable,
    CS_Normal,
    CS_Reverse,
} E_ColourStatus;

/*
    Globals
*/
static int	 	        colourFormattingOpt = 0;
static E_ColourStatus 	colourFormattingStatus = CS_Disable;

void displayFormatted (bool MetaFlag, char *Text, ...);

/*==============================================================================
Function: main
Params:
p1          - Options
p2          - filename to dump
Return:
0 if ok, 1 if error
Description:
See module description
==============================================================================*/
int main(int argc, char **argv)
{
    int     i,
            c,
            width = LONGS_PER_LINE_ORD * sizeof (long),
            bytes = sizeof (long),
            itemsRead,
            nextFile,
            pos = 0,
            offsetSt = 0,
            offsetEnd = INT_MAX_OFFS;

    bool    nzFlag,
            nzprintFlag,
            skipzFlag = false,
            asciiFlag = true,
            hexFlag = true,
            resetFlag = false,
            displayNameFlag = false,
            exitStatusFlag = true;

    char    pntr[LONGS_PER_LINE_WIDE * sizeof (long)],
            printstr[MAX_PS_SIZE],
            *psSizeStr = PS_LONG,
            *psZeroStr = PS_NZ,
            *filler = FI_LONG,
            unprintableChar = '.';

    FILE    *file,
            *helpOut = stderr;

    /*
    * Process arguments
    */
    while ((c = getopt (argc, argv, FLAGS)) != EOF)
    {
        switch ((char) c)
        {
        case 'b' :
            bytes = sizeof (char);
            psSizeStr = PS_BYTE;
            filler = FI_BYTE;
            break;

        case 's' :
            bytes = sizeof (short);
            psSizeStr = PS_SHORT;
            filler = FI_SHORT;
            break;

        case 'l' :
            bytes = sizeof (long);
            psSizeStr = PS_LONG;
            filler = FI_LONG;
            break;

        case 'w' :
            width = LONGS_PER_LINE_WIDE * sizeof (long);
            break;

        case 'n' :
            width = LONGS_PER_LINE_NARROW * sizeof (long);
            break;

        case 'm' :
            width = LONGS_PER_LINE_ORD * sizeof (long);
            break;

        case 'o' :
            if (*optarg == ':')
            {
                sscanf (optarg, ":%x", &offsetEnd);
            }
            else if (isdigit (*optarg))
            {
                sscanf (optarg, "%x:%x", &offsetSt, &offsetEnd);
            }
            else
            {
                fprintf (stderr, "Invalid offset: %s\n", optarg);
                exit (1);
            }
            break;

        case 'a' :
            asciiFlag = true;
            hexFlag = false;
            break;

        case 'x' :
            hexFlag = true;
            asciiFlag = false;
            break;

        case 'f' :
            hexFlag = true;
            asciiFlag = true;
            break;

        case 'z' :
            skipzFlag = true;
            break;

        case 'Z' :
            skipzFlag = false;
            break;

        case 'd' :
            displayNameFlag = true;
            break;

        case 'D' :
            displayNameFlag = false;
            break;

        case 'r' :
            resetFlag = true;
            break;

        case 'R' :
            resetFlag = false;
            break;

        case 'c' :
            if ((optarg == NULL) || *optarg == 'n')
            {
                colourFormattingOpt = 0;
                colourFormattingStatus = CS_Disable;
            }
            else
            {
                colourFormattingOpt = -1;
                for (i = 0; i < MaxColours; i++)
                {
                    if (Colours[i].charOption == tolower (*optarg))
                    {
                        colourFormattingOpt = i;
                        break;
                    }
                }
                if (colourFormattingOpt == -1)
                {
                    fprintf (stderr, "Invalid colour formatting option: %c\n", *optarg);
                    exit (1);
                }
                colourFormattingStatus = *optarg >= 'a' ? CS_Normal : CS_Reverse;
            }
            break;

        case 'u' :
            if (optarg == NULL)
            {
               unprintableChar = ' ';
            }
            else
            {
                unprintableChar = *optarg;
            }
            break;

        case '0' :
            psZeroStr = PS_ZE;
            break;

        case '1' :
            psZeroStr = PS_NZ;
            break;

        case 'v' :
            printf ("%s\n", VERSION);
            break;

        case 'h' :
            exitStatusFlag = false;
            helpOut = stdout;
            fprintf (helpOut, "%s\n", VERSION);

        default :
            fprintf (helpOut, "Dump a file to the terminal in hex and ASCII format.\n");
            fprintf (helpOut, "%s [-%s] [-] [<filename to dump>] [...]\n", argv[0], HELP_FLAGS);
            fprintf (helpOut, "\t-b Byte.     Display in sets of 1 byte\n");
            fprintf (helpOut, "\t-s Short.    Display in sets of 2 bytes\n");
            fprintf (helpOut, "\t-l Long.     Display in sets of 4 bytes (default)\n");
            fprintf (helpOut, "\t-n Narrow.   Display %d bytes per line\n", LONGS_PER_LINE_NARROW * sizeof (long));
            fprintf (helpOut, "\t-m Medium    Display %d bytes per line (default)\n", LONGS_PER_LINE_ORD * sizeof (long));
            fprintf (helpOut, "\t-w Wide.     Display %d bytes per line\n", LONGS_PER_LINE_WIDE * sizeof (long));
            fprintf (helpOut, "\t-a Ascii.    Display ascii only\n");
            fprintf (helpOut, "\t-x Hex.      Display hex only\n");
            fprintf (helpOut, "\t-f Full.     Display both hex and ascii (default)\n");
            fprintf (helpOut, "\t-0 0-pad.    Pad with leading zeros (Number 0)\n");
            fprintf (helpOut, "\t-1 0-pad*.   Don't pad with leading zeros (number 1)\n");
            fprintf (helpOut, "\t-z Zero.     Skip display A of lines containing only zero (NULL) bytes\n");
            fprintf (helpOut, "\t-Z Zero*.    Don't skip display A of lines containing only zero (NULL) bytes\n");
            fprintf (helpOut, "\t-d Display.  Display the name of each file before the output\n");
            fprintf (helpOut, "\t-D Display*. Don't display the name of each file before the output\n");
            fprintf (helpOut, "\t-r Reset.    Reset the byte count for each file dumped\n");
            fprintf (helpOut, "\t-R Reset*.   Don't reset the byte count for each file dumped\n");
            fprintf (helpOut, "\t-o<s>:<e>    Starts and ends the dump from the specified offsets\n");
            fprintf (helpOut, "\t-c[<char>].  Set formatting (See note)\n");
            fprintf (helpOut, "\t-u<char>     In ascii mode, specifies the unprintable character. See note\n");
            fprintf (helpOut, "\t-h Help.     Print this help\n");
            fprintf (helpOut, "\t-v Vers.     Print version number\n");
            fprintf (helpOut, "\nIf no filename is supplied or filename is - then stdin is used.\n");
            fprintf (helpOut, "If more than one filename is supplied then each is processed in turn.\n");
            fprintf (helpOut, "\nNote that -1 turns off -0 and similarly the options -R, -D and -Z turn off their"
                "\ncounterparts -r, -d and -z\n");
            fprintf (helpOut, "\nThe option -u must be on its own (eg -u -z or -u* -z). By default a . is"
                "\ndisplayed to represent unprintable characters. -u on its own uses a space,"
                "\notherwise the specified character is used.\n");
            fprintf (helpOut, "\nThe option -c must be on its own (eg -c -z or -cr -z). -c on its own disables"
                "\nformatting (as does -cn). The format defines how all characters not part of the binary"
                "\n(including filenames and ascii unprintable characters) are displayed. The uppercase character"
                "\nreverses this. Allowable characters are:"
                "\n\teE - Emphasis (black)"
                        "\n\trR - Red"
                        "\n\tgG - Green"
                        "\n\tyY - Yellow"
                        "\n\tbB - Blue"
                        "\n\tmM - Magenta"
                        "\n\tcC - Cyan"
                        "\n\twW - White\n");
            fprintf (helpOut, "\nThe option -o specifies the start and end offsets for the dump (in hex). By default"
                "\ndumping is from the start of the file to the end. The start and end are separated by :,"
                "\neg -o1000:4400. Either start or end can be left out. eg -o:FFFF, -o1000.\n");
            exit (exitStatusFlag);
            break;
        }
    } /* end while */

    sprintf (printstr, "%s%s", psZeroStr, psSizeStr);
    nextFile = optind;
    do
    {
        if ((nextFile >= argc) || (*(argv[nextFile]) == '-'))
        {
            file = stdin;
            if (displayNameFlag == true)
            {
                displayFormatted (true, "FILE: stdin\n");
            }
        }
        else
        {
            file = fopen (argv[nextFile], "r");
            if (file == NULL)
            {
                fprintf (stderr, "Can't open file %s\n", argv[nextFile]);
                exit (2);
            }
            if (displayNameFlag == true)
            {
                displayFormatted (true, "FILE: %s\n", argv[nextFile]);
            }

        }
        if (resetFlag == true)
        {
            pos = 0;
        }

        while ((itemsRead = fread (pntr, 1, width, file)) > 0)
        {
            if (pos < offsetSt - width + 1)
            {
                pos += width;
                continue;
            }
            if (pos >= offsetEnd + 1)
            {
                break;
            }
            if (skipzFlag)
            {
                nzFlag = false;
                for (i = 0; i < itemsRead; i++)
                {
                    if (pntr[i] != 0)
                    {
                        nzFlag = true;
                        nzprintFlag = false;
                        break;
                    }
                }
            }

            if (skipzFlag && !nzFlag)
            {
                if (!nzprintFlag)
                {
                    displayFormatted (true, ".... Skipping zero bytes\n");
                    nzprintFlag = true;
                }
                pos += width;
                continue;
            }
            displayFormatted (true, "%04X: ", pos);

            if (hexFlag)
            {
                long    prTmp;

                for (i = 0; i < itemsRead; i += bytes)
                {
                    switch (bytes)
                    {
                    case sizeof (char) :
                        {
                            unsigned char tmp = pntr[i];
                            prTmp = tmp;
                        }
                        break;

                    case sizeof (short) :
                        {
                            unsigned short *tmp = (unsigned short *) &pntr[i];
                            prTmp = *tmp;
                        }
                        break;

                    case sizeof (long) :
                        {
                            unsigned long *tmp = (unsigned long *) &pntr[i];
                            prTmp = *tmp;
                        }
                        break;
                    }
                    displayFormatted (false, printstr, prTmp);
                }
                for (i = 0; i < (width - itemsRead - bytes + 1); i += bytes)
                {
                    displayFormatted (false, "%s", filler);
                }
                displayFormatted (false, "   ");
            }
            if (asciiFlag)
            {
                for (i = 0; i < itemsRead; i++)
                {
                    if (isprint (pntr[i]))
                    {
                        displayFormatted (false, "%c", pntr[i]);
                    }
                    else
                    {
                        displayFormatted (true, "%c", unprintableChar);

                    }
                }
            }
            pos += width;
            memset (pntr, 0, sizeof (pntr));
            printf ("\n");
        }
        fclose (file);
    } while (++nextFile < argc);
    exit (0);
}

/*=================================================================================================
Function: DisplayFormatted
        Displays the text as formattted (eg bold, red etc).
char *FormatStr	- printf style format string
...		- Optional formatting characters
==================================================================================================*/
void displayFormatted (bool MetaFlag, char *FormatStr, ...)
{
    va_list ap;
    va_start (ap, FormatStr);

    if (colourFormattingStatus != CS_Disable)
    {
       if (!MetaFlag)
        {
            if (colourFormattingStatus == CS_Reverse)
            {
                printf ("%s", Colours[colourFormattingOpt].escSeq);
            }
        }
        else
        {
            if (colourFormattingStatus == CS_Normal)
            {
                printf ("%s", Colours[colourFormattingOpt].escSeq);
            }
        }
        vprintf (FormatStr, ap);
        printf ("%s", D_RESET);
    }
    else
    {
        vprintf (FormatStr, ap);
    }
}

