/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.exceptions.NucleusUserException;

/**
 * An expression representing a subquery.
 *
 * @version $Revision: 1.6 $
 **/
public class SubqueryExpression extends BooleanExpression
{
    /** Subquery expression. */
    private final QueryExpression subqueryExpr;

    /**
     * Construct an expression containing this subquery.
     * @param qs the QueryExpression
     * @param expr Expression for the subquery
     */
    public SubqueryExpression(QueryExpression qs, QueryExpression expr)
    {
        super(qs);

        this.subqueryExpr = expr;

        // Mark the subquery as having our statement as its parent
        expr.setParent(qs);

        st.append("(").append(expr).append(')');
    }

    /**
     * Convenience accessor for the subquery expression.
     * @return Subquery expression
     */
    public QueryExpression getSubquery()
    {
        return subqueryExpr;
    }

    /**
     * Method to convert the subquery statement into ALL ({subquery})
     */
    public void all()
    {
        // TODO This is RDBMS-specific and shouldn't be in here
        st.clearStatement();
        st.append("ALL (").append(subqueryExpr).append(")");
    }

    /**
     * Method to convert the subquery statement into ANY ({subquery})
     */
    public void any()
    {
        // TODO This is RDBMS-specific and shouldn't be in here
        st.clearStatement();
        st.append("ANY (").append(subqueryExpr).append(")");
    }

    /**
     * Method to convert the subquery statement into EXISTS ({subquery})
     */
    public void exists()
    {
        // TODO This is RDBMS-specific and shouldn't be in here
        st.clearStatement();
        st.append("EXISTS (").append(subqueryExpr).append(")");
    }

    /**
     * Method to convert the subquery statement into "expr IN ({subquery})"
     * @param expr The expression that is being contained
     * @return BooleanExpression for the IN
     */
    public BooleanExpression in(ScalarExpression expr)
    {
        return new InExpression(expr, OP_IN, this);
    }

    /**
     * Method to return an expression for whether the specified expression is contained in this subquery. 
     * Uses IN operator.
     * @param expr The expression that is being contained
     * @return BooleanExpression for the contains(expr)
     */
    public BooleanExpression containsMethod(ScalarExpression expr)
    {
        return in(expr);
    }

    /**
     * Method to return the size of the subquery.
     * @return Size of the subquery
     */
    public NumericExpression sizeMethod()
    {
        throw new NucleusUserException("size() is not currently supported on subqueries. Use COUNT(*) in the subquery");
    }

    /**
     * Method to return an expression for whether the subquery is empty (no results).
     * @return BooleanExpression for the isEmpty()
     */
    public BooleanExpression isEmptyMethod()
    {
        throw new NucleusUserException("isEmpty() is not currently supported on subqueries. Use COUNT(*) in the subquery and compare with 0");
    }
}