/**********************************************************************
Copyright (c) 2003 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.List;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of an expression with an SQL time type (SQLDate, SQLTime,
 * SQLTimestamp)
 *
 * @version $Revision: 1.10 $
 **/
public class SqlTemporalExpression extends ScalarExpression
{
    protected SqlTemporalExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * 
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */    
    public SqlTemporalExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    /**
     * Generates statement as "FUNCTION_NAME"
     * @param functionName
     * @param qs QueryExpression
     */    
    public SqlTemporalExpression(String functionName, QueryExpression qs)
    {
        super(qs);
        st.append(functionName);
    }

    /**
     * Generates statement as e.g. FUNCTION_NAME(arg[,argN])
     * @param functionName
     * @param args ScalarExpression list
     */    
    public SqlTemporalExpression(String functionName, List args)
    {
        super(functionName, args);
    }

    /**
     * Performs a function on two arguments.
     * op(operand1,operand2)
     * operand1 op operand2 
     * @param operand1 the first expression
     * @param op the operator between operands
     * @param operand2 the second expression
     */    
    public SqlTemporalExpression(ScalarExpression operand1, DyadicOperator op, ScalarExpression operand2)
    {
        super(operand1, op, operand2);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.eq(this);
        }
        else if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.noteq(this);
        }
        else if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);
        }
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);
        }
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);
        }
        else
        {
            return super.gteq(expr);
        }
    }

    public BooleanExpression in(ScalarExpression expr)
    {
        return new BooleanExpression(this, OP_IN, expr);
    }

    /**
     * Method to return an expression for the day of the month (for this date).
     * @return The expression for the day of the month.
     **/
    public NumericExpression getDayMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("day", this);
    }

    /**
     * Method to return an expression for the month (for this date).
     * @return The expression for the month.
     **/
    public NumericExpression getMonthMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("month", this);
    }

    /**
     * Method to return an expression for the year (for this date).
     * @return The expression for the year.
     **/
    public NumericExpression getYearMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("year", this);
    }

    /**
     * Method to return an expression for the hour (for this time).
     * @return The expression for the hour.
     **/
    public NumericExpression getHourMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("hour", this);
    }

    /**
     * Method to return an expression for the minute (for this time).
     * @return The expression for the month.
     **/
    public NumericExpression getMinuteMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("minute", this);
    }

    /**
     * Method to return an expression for the second (for this time).
     * @return The expression for the second.
     **/
    public NumericExpression getSecondMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("second", this);
    }
}