# Part of the A-A-P recipe executive: Setup using GCC

# Copyright (c) 2002-2003 stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

#
# This module sets up variables and actions for using the GCC compiler tools.
# Last Change: 2005 Nov 30
#

from RecPython import *
import Global
from Action import action_add
from Dictlist import str2dictlist
from RecPos import RecPos


def exists():
    """
    Return TRUE when the GCC toolchain can be found.
    """
    # TODO: "cc" may also be gcc, use the gcc configure check for this.
    return program_path("gcc")


def define_actions():
    """
    Define the actions that GCC can accomplish.
    """
    # Allow the user to define the gcc command to be used with $GCC.
    rd = Global.globals
    if not rd.get("GCC"):
        rd["GCC"] = "gcc"

    # When "g++" or "gxx" can be found use it for compiling C++.  When it
    # cannot be found use "gcc" (causes problems when linking?).
    if not rd.get("GXX"):
        if program_path("g++"):
            rd["GXX"] = "g++"
        elif program_path("gxx"):
            rd["GXX"] = "gxx"
        else:
            rd["GXX"] = "gcc"

    define_action("compile_gcc", 0, """
	    :buildcheck $OPTIMIZE $?DEBUG
            :sys $GCC $CPPFLAGS $?DEFINE $?INCLUDE `cflags_normal()` $CFLAGS -o $target -c $source
            """,
            outtypes = ["object", "libobject"],
            intypes = ["c"])

    define_action("compile_gcc", 0, """
	    :buildcheck $OPTIMIZE $?DEBUG
            :sys $GCC $CPPFLAGS $?DEFINE $?INCLUDE `cflags_normal()` $CFLAGS -fPIC -o $target -c $source
            """,
            outtypes = ["dllobject"],
            intypes = ["c"])

    define_action("compile_gxx", 0, """
            :buildcheck $OPTIMIZE $?DEBUG
            :attr {buildaction = build_gxx} $target	# Need to build with $CXX
            :sys $GXX $CPPFLAGS $?DEFINE $?INCLUDE `cflags_normal()` $CXXFLAGS -o $target -c $source
            """,
            outtypes = ["object", "libobject"],
            intypes = ["cpp"])

    define_action("compile_gxx", 0, """
	    :buildcheck $OPTIMIZE $?DEBUG
            :sys $GXX $CPPFLAGS $?DEFINE $?INCLUDE `cflags_normal()` $CXXFLAGS -fPIC -o $target -c $source
            """,
            outtypes = ["dllobject"],
            intypes = ["cpp"])

    define_action("build_gcc", 0, """
            :sys $GCC $LDFLAGS `cflags_normal()` -o $target $source $?LIBS
            """,
            outtypes = ["default"],
            intypes = ["object", "libobject"])

    define_action("build_gxx", 0, """
            :sys $GXX $LDFLAGS `cflags_normal()` -o $target $source $?LIBS
            """,
            outtypes = ["default"],
            intypes = ["object", "libobject"])


def use_actions(scope):
    """
    Setup variables so that the default actions use the GCC actions.
    """
    scope["C_COMPILE_ACTION"] = "compile_gcc"
    scope["CXX_COMPILE_ACTION"] = "compile_gxx"
    scope["C_BUILD_ACTION"] = "build_gcc"
    scope["CXX_BUILD_ACTION"] = "build_gxx"

    # Attempt using gcc for dependency checks.
    scope["HASGCC"] = ""
    scope["HASGCCXX"] = ""

# vim: set sw=4 et sts=4 tw=79 fo+=l:
