//
//  Controller.m
//  Cosmic Debris
//
//  Created by John Schilling on 11/22/04.
//  Copyright 2004 John Schilling. All rights reserved.
//

#import "Controller.h"
#import <Message/NSMailDelivery.h>

#define STATUS_ICON_ANIMATION_SPEED_SECONDS     0.050

static Controller *sharedInstance = nil;

@implementation Controller

#pragma mark
#pragma mark PRIVATE INSTANCE METHODS
#pragma mark

- (id)init
{
    if (sharedInstance) {
	NSLog(@"Attempt to allocate more than one AppController object!");
	return nil;
    } else {
	[super init];
	sharedInstance          = self;
        _statusDLIcons          = [[NSMutableArray alloc] init];
        _lastUpdatedString      = [[NSMutableString alloc] init];
        _nextUpdateString       = [[NSMutableString alloc] init];
        _alertSoundName         = [[NSMutableString alloc] init];
        _emailAddress           = [[NSMutableString alloc] init];
        _lastOvalImageUpdate    = [[NSMutableString alloc] init];
        _ovalFileName           = [[NSString stringWithString:@"auroraOvalNorth.gif"] retain];
        _lastKPImageUpdate      = [[NSMutableString alloc] init];
        _KPFileName             = [[NSString stringWithString:@"planetaryKPIndex.gif"] retain];
        _lastAppPath            = [[NSMutableString alloc] init];
        [self setupCacheFolder];
    }
    return sharedInstance;
}

+ (Controller *)sharedInstance
{
    return sharedInstance ? sharedInstance : [[self alloc] init];
}

- (void)awakeFromNib
{
    _appNameString           = [[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleExecutable"] retain];
    _versionString           = [[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleVersion"] retain];
    [_aboutBoxenVersionNumberField setStringValue:[NSString stringWithFormat:@"Version Number %@", _versionString]];
    [_prefsWindowVersionNumberField setStringValue:[NSString stringWithFormat:@"%@ %@ (c) 2004-2008 John Schilling & StimpSoft Inc.", _appNameString, _versionString]];
    
    [self loadPreferences];
    [self compileSystemSoundsMenu];
    [self loadCachedOvalImage];
    [self loadCachedKPImage];
    
    [self setupPreferencesPanel];
    
    _STDDownloader  = [[Downloader alloc] initWithURL:@"http://www.spacew.com/index.php" delegate:self];
    _PCAFDownloader = [[Downloader alloc] initWithURL:@"http://www.sec.noaa.gov/ftpdir/latest/RSGA.txt" delegate:self];
    _ovalDownloader = [[Downloader alloc] initWithURL:@"http://www.sec.noaa.gov/pmap/gif/pmapN.gif" delegate:self];
    _KPDownloader   = [[Downloader alloc] initWithURL:@"http://www.sec.noaa.gov/rt_plots/Kp.gif" delegate:self];
    
    [self loadDownloadingIcons];
    [self initStatusItem];
    [self updateCosmicMenu];
}

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
    if (_restoreWindowPositions)
    {
        if (_ovalWindowOpen) {
            [_ovalWindow open:self];
            if (_ovalWindowDocked) [_ovalWindow performMiniaturize:self];
        }
        if (_KPWindowOpen) {
            [_KPWindow open:self];
            if (_KPWindowDocked) [_KPWindow performMiniaturize:self];
        }
    }
    
    [self checkDownloadAtStartup];
    [self startUpdateTimer];
}

- (void)applicationWillTerminate:(NSNotification *)aNotification
{
    if (_STDDownloader)   [_STDDownloader cleanup];
    if (_PCAFDownloader)  [_PCAFDownloader cleanup];
    if (_ovalDownloader)  [_ovalDownloader cleanup];
    if (_KPDownloader)    [_KPDownloader cleanup];

    [self stopUpdateTimer];
    [self stopAnimationTimer];
    [self savePreferences];
}

- (void)applicationDidHide:(NSNotification *)aNotification
{
    /* NOP */
}

- (BOOL)validateMenuItem:(NSMenuItem *)anItem
{
    if (anItem == _displayOvalImageMenuItem) {
        // Display current oval image menu item. Disable if not auto downloading this image.
        return _downloadOvalImage;
    } else if (anItem == _displayKPImageMenuItem) {
        // Display current oval image menu item. Disable if not auto downloading this image.
        return _downloadKPImage;
    }
        
    return YES;
}

- (void)mouseDown:(NSEvent *)theEvent
{
    [[self window] makeFirstResponder:[[self window] contentView]];
}

- (void)dealloc
{
    if (_STDDownloader)  [_STDDownloader release];
    _STDDownloader = nil;
    if (_PCAFDownloader) [_PCAFDownloader release];
    _PCAFDownloader = nil;
    if (_ovalDownloader) [_ovalDownloader release];
    _ovalDownloader = nil;
    if (_KPDownloader)   [_KPDownloader release];
    _KPDownloader = nil;
    
    [_statusItem release];
    
    [_statusDLIcons removeAllObjects];
    [_statusDLIcons release];
    
    [_lastUpdatedString release];
    [_lastUpdatedDate release];
    [_nextUpdateString release];
    [_nextUpdateDate release];
    [_alertSoundName release];
    [_emailAddress release];
    
    [_cacheFolderPath release];
    
    [_ovalFileName release];
    [_lastOvalImageUpdate release];
    [_KPFileName release];
    [_lastKPImageUpdate release];
    
    [_appNameString release];
    [_versionString release];
    
    [_lastAppPath release];
    _lastAppPath = nil;
    
    [super dealloc];
}









#pragma mark
#pragma mark PREFERNCE METHODS
#pragma mark

- (void)loadPreferences
{
    NSDictionary *appDefaults = [NSDictionary dictionaryWithObjectsAndKeys: 
                                    @"0",     @"_highLatitudes", 
                                    @"0",     @"_middleLatitudes",
                                    @"0",     @"_lowLatitudes", 
                                    @"0",     @"_PCAFlevel", 
                                    @"0",     @"_currentAEItem", 
                                    @"",      @"_lastUpdate", 
                                    @"1",     @"_autoUpdates", 
                                    @"2",     @"_autoUpdateRate", 
                                    @"1",     @"_downloadOvalImage", 
                                    @"Unknown", @"_lastOvalImageUpdate", 
                                    @"0",     @"_ovalWindowOpen", 
                                    @"0",     @"_ovalWindowDocked", 
                                    @"1",     @"_downloadKPImage", 
                                    @"Unknown", @"_lastKPImageUpdate", 
                                    @"0",     @"_KPWindowOpen", 
                                    @"0",     @"_KPWindowDocked", 
                                    @"1",     @"_restoreWindowPositions", 
                                    @"1",     @"_downloadAtStartup", 
                                    @"1",     @"_alertOnChange", 
                                    @"0",     @"_alertOnAllItemChanges", 
                                    @"1",     @"_alertThreshold", 
                                    @"1",     @"_alertWithSound", 
                                    @"Beep",  @"_alertSoundName", 
                                    @"0",     @"_alertWithMessage", 
                                    @"0",     @"_autoCloseAlertWindow", 
                                    @"0",     @"_alertWithEmail", 
                                    @"0",     @"_emailShortMessages", 
                                    @"",      @"_emailAddress", 
                                    @"60.0",  @"_timeout", 
                                    @"0",     @"_statusItemStyle", 
                                    nil];
                                    
    [[NSUserDefaults standardUserDefaults] registerDefaults:appDefaults];
    
    _timeout                = [[NSUserDefaults standardUserDefaults] floatForKey:@"_timeout"];
    _currentHighLat         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_highLatitudes"];
    _currentMidLat          = [[NSUserDefaults standardUserDefaults] integerForKey:@"_middleLatitudes"];
    _currentLowLat          = [[NSUserDefaults standardUserDefaults] integerForKey:@"_lowLatitudes"];
    _currentPCAFlevel       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_PCAFlevel"];
    _previousHighLat        = _currentHighLat;
    _previousMidLat         = _currentMidLat;
    _previousLowLat         = _currentLowLat;
    _previousPCAFlevel      = _currentPCAFlevel;
    _currentAEItem          = [[NSUserDefaults standardUserDefaults] integerForKey:@"_currentAEItem"];
    _autoUpdates            = [[NSUserDefaults standardUserDefaults] integerForKey:@"_autoUpdates"];
    _autoUpdateRate         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_autoUpdateRate"];
    _downloadAtStartup      = [[NSUserDefaults standardUserDefaults] integerForKey:@"_downloadAtStartup"];
    
    _downloadOvalImage      = [[NSUserDefaults standardUserDefaults] integerForKey:@"_downloadOvalImage"];
    [_lastOvalImageUpdate setString:[[NSUserDefaults standardUserDefaults] objectForKey:@"_lastOvalImageUpdate"]];
    _ovalWindowOpen         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_ovalWindowOpen"];
    _ovalWindowDocked       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_ovalWindowDocked"];
    _downloadKPImage        = [[NSUserDefaults standardUserDefaults] integerForKey:@"_downloadKPImage"];
    [_lastKPImageUpdate setString:[[NSUserDefaults standardUserDefaults] objectForKey:@"_lastKPImageUpdate"]];
    _KPWindowOpen           = [[NSUserDefaults standardUserDefaults] integerForKey:@"_KPWindowOpen"];
    _KPWindowDocked         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_KPWindowDocked"];
    _restoreWindowPositions = [[NSUserDefaults standardUserDefaults] integerForKey:@"_restoreWindowPositions"];
    
    _alertOnChange          = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertOnChange"];
    _alertOnAllItemChanges  = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertOnAllItemChanges"];
    _alertThreshold         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertThreshold"];
    _alertWithSound         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertWithSound"];
    [_alertSoundName setString:[[NSUserDefaults standardUserDefaults] objectForKey:@"_alertSoundName"]];
    _alertWithMessage       = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertWithMessage"];
    _autoCloseAlertWindow   = [[NSUserDefaults standardUserDefaults] integerForKey:@"_autoCloseAlertWindow"];
    _alertWithEmail         = [[NSUserDefaults standardUserDefaults] integerForKey:@"_alertWithEmail"];
    _emailShortMessages     = [[NSUserDefaults standardUserDefaults] integerForKey:@"_emailShortMessages"];
    [_emailAddress setString:[[NSUserDefaults standardUserDefaults]  objectForKey:@"_emailAddress"]];
    
    _statusItemStyle        = [[NSUserDefaults standardUserDefaults]  integerForKey:@"_statusItemStyle"];
    
    
    
    
    NSString *lastUpdated = [[NSUserDefaults standardUserDefaults] objectForKey:@"_lastUpdate"];
    if (lastUpdated && [lastUpdated length] != 0) {
        _lastUpdatedDate = [[NSDate dateWithNaturalLanguageString:lastUpdated] retain];
    } else {
        _lastUpdatedDate = [[NSCalendarDate calendarDate] retain];
    }
    [_lastUpdatedString setString:[_lastUpdatedDate descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    [self calculateNextUpdate];
}

- (void)savePreferences
{
    [[NSUserDefaults standardUserDefaults] setFloat:_timeout forKey:@"_timeout"];
    [[NSUserDefaults standardUserDefaults] setInteger:_currentHighLat forKey:@"_highLatitudes"];
    [[NSUserDefaults standardUserDefaults] setInteger:_currentMidLat forKey:@"_middleLatitudes"];
    [[NSUserDefaults standardUserDefaults] setInteger:_currentLowLat forKey:@"_lowLatitudes"];
    [[NSUserDefaults standardUserDefaults] setInteger:_currentPCAFlevel forKey:@"_PCAFlevel"];
    [[NSUserDefaults standardUserDefaults] setInteger:_currentAEItem forKey:@"_currentAEItem"];
    [[NSUserDefaults standardUserDefaults] setInteger:_autoUpdates forKey:@"_autoUpdates"];
    [[NSUserDefaults standardUserDefaults] setInteger:_autoUpdateRate forKey:@"_autoUpdateRate"];
    [[NSUserDefaults standardUserDefaults] setInteger:_downloadAtStartup forKey:@"_downloadAtStartup"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_downloadOvalImage forKey:@"_downloadOvalImage"];
    [[NSUserDefaults standardUserDefaults] setObject:_lastOvalImageUpdate forKey:@"_lastOvalImageUpdate"];
    [[NSUserDefaults standardUserDefaults] setInteger:_ovalWindowOpen forKey:@"_ovalWindowOpen"];
    [[NSUserDefaults standardUserDefaults] setInteger:_ovalWindowDocked forKey:@"_ovalWindowDocked"];
    [[NSUserDefaults standardUserDefaults] setInteger:_downloadKPImage forKey:@"_downloadKPImage"];
    [[NSUserDefaults standardUserDefaults] setObject:_lastKPImageUpdate forKey:@"_lastKPImageUpdate"];
    [[NSUserDefaults standardUserDefaults] setInteger:_KPWindowOpen forKey:@"_KPWindowOpen"];
    [[NSUserDefaults standardUserDefaults] setInteger:_KPWindowDocked forKey:@"_KPWindowDocked"];
    [[NSUserDefaults standardUserDefaults] setInteger:_restoreWindowPositions forKey:@"_restoreWindowPositions"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_alertOnChange forKey:@"_alertOnChange"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertOnAllItemChanges forKey:@"_alertOnAllItemChanges"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertThreshold forKey:@"_alertThreshold"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertWithSound forKey:@"_alertWithSound"];
    [[NSUserDefaults standardUserDefaults] setObject:_alertSoundName forKey:@"_alertSoundName"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertWithMessage forKey:@"_alertWithMessage"];
    [[NSUserDefaults standardUserDefaults] setInteger:_autoCloseAlertWindow forKey:@"_autoCloseAlertWindow"];
    [[NSUserDefaults standardUserDefaults] setInteger:_alertWithEmail forKey:@"_alertWithEmail"];
    [[NSUserDefaults standardUserDefaults] setInteger:_emailShortMessages forKey:@"_emailShortMessages"];
    [[NSUserDefaults standardUserDefaults] setObject:_emailAddress forKey:@"_emailAddress"];
    
    [[NSUserDefaults standardUserDefaults] setInteger:_statusItemStyle forKey:@"_statusItemStyle"];
    
    NSString *lastUpdated = [_lastUpdatedDate descriptionWithCalendarFormat:@"%B %d %Y at %I:%M %p"];
    [[NSUserDefaults standardUserDefaults] setObject:lastUpdated forKey:@"_lastUpdate"];
    
    [[NSUserDefaults standardUserDefaults] synchronize];
}

- (void)setupPreferencesPanel
{
    [_downloadAtStartupButton setState:(_downloadAtStartup ? NSOnState : NSOffState)];
    [_autoUpdatesButton setState:(_autoUpdates ? NSOnState : NSOffState)];
    [_autoUpdateRateMenu selectItemAtIndex:_autoUpdateRate];
    [_autoUpdateRateMenu setEnabled:_autoUpdates];
    
    [_downloadOvalImageButton setState:(_downloadOvalImage ? NSOnState : NSOffState)];
    [_downloadKPImageButton setState:(_downloadKPImage ? NSOnState : NSOffState)];
    [_restoreWindowPositionsButton setState:(_restoreWindowPositions ? NSOnState: NSOffState)];
    
    [_currentAEItemMenu selectItemAtIndex:_currentAEItem];
    [_timeoutTextField setIntValue:_timeout];
    [_timeoutStepper setIntValue:_timeout];

    [_alertOnChangeButton setState:(_alertOnChange ? NSOnState : NSOffState)];
    [_alertOnAllChangesButton setEnabled:_alertOnChange];
    [_alertOnAllChangesButton setState:_alertOnAllItemChanges];
    [_alertThresholdMenu setEnabled:_alertOnChange];
    [_alertThresholdMenu selectItemAtIndex:_alertThreshold];
    [_alertWithSoundButton setEnabled:_alertOnChange];
    [_alertWithSoundButton setState:(_alertWithSound ? NSOnState : NSOffState)];
    [_alertSoundsMenu setEnabled:((_alertOnChange && _alertWithSound) ? YES : NO)];
    [_alertWithMessageButton setEnabled:_alertOnChange];
    [_alertWithMessageButton setState:(_alertWithMessage ? NSOnState : NSOffState)];
    [_autoCloseAlertWindowButton setState:_alertWithMessage];
    [_autoCloseAlertWindowButton setEnabled:((_alertOnChange && _alertWithMessage) ? YES : NO)];
    [_alertWithEmailButton setEnabled:_alertOnChange];
    [_alertWithEmailButton setState:(_alertWithEmail ? NSOnState : NSOffState)];
    [_emailShortMessagesButton setState:_emailShortMessages];
    [_emailShortMessagesButton setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
    [_emailAddressField setStringValue:_emailAddress];
    [_emailAddressField setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
    [_statusItemStyleMenu selectItemAtIndex:_statusItemStyle];
    [_isLoginItemButton setState:([self isLoginItem] ? NSOnState : NSOffState)];
}

- (IBAction)openPreferencesWindow:(id)sender
{
    [self setLastApplicationPath];
    [NSApp activateIgnoringOtherApps:YES];
    [_prefsWindow center];
    [_prefsWindow makeKeyAndOrderFront:self];
    [[self window] makeFirstResponder:[[self window] contentView]];
}

- (IBAction)setDownloadsAtStartup:(id)sender
{
    _downloadAtStartup = [sender state];
}

- (IBAction)setAutoUpdates:(id)sender
{
    _autoUpdates = [sender state];
    [_autoUpdateRateMenu setEnabled:_autoUpdates];
    [self updateCosmicMenu];
}

- (IBAction)setAutoUpdateRate:(id)sender
{
    _autoUpdateRate = [[sender selectedItem] tag];
    [self calculateNextUpdate];
    [self updateCosmicMenu];
}

- (IBAction)setDownloadsOvalImage:(id)sender
{
    _downloadOvalImage = [sender state];
}

- (IBAction)setDownloadsKPImage:(id)sender
{
    _downloadKPImage = [sender state];
}

- (IBAction)setRestoreWindowPositions:(id)sender
{
    _restoreWindowPositions = [sender state];
}

- (IBAction)setAlertOnChange:(id)sender
{
    _alertOnChange = [sender state];
    [_alertThresholdMenu setEnabled:_alertOnChange];
    [_alertWithSoundButton setEnabled:_alertOnChange];
    [_alertSoundsMenu setEnabled:((_alertOnChange && _alertWithSound) ? YES : NO)];
    [_alertWithMessageButton setEnabled:_alertOnChange];
    [_autoCloseAlertWindowButton setEnabled:((_alertOnChange && _alertWithMessage) ? YES : NO)];
    [_alertOnAllChangesButton setEnabled:_alertOnChange];
    [_alertWithEmailButton setEnabled:_alertOnChange];
    [_emailShortMessagesButton setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
    [_emailAddressField setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
}

- (IBAction)setAlertOnAllChanges:(id)sender
{
    _alertOnAllItemChanges = [sender state];
}

- (IBAction)setAlertThreshold:(id)sender
{
    _alertThreshold = [[sender selectedItem] tag];
}

- (IBAction)setAlertWithSound:(id)sender
{
    _alertWithSound = [sender state];
    [_alertSoundsMenu setEnabled:((_alertOnChange && _alertWithSound) ? YES : NO)];
}

- (IBAction)setAlertSoundName:(id)sender
{
    NSString *sndName = [[sender selectedItem] title];
    if (![sndName isEqualToString:@"Beep"]) {
        NSSound *snd = [NSSound soundNamed:sndName];
        if (snd) {
            [_alertSoundName setString:sndName];
            [snd play];
        } else {
            [_alertSoundName setString:@"Beep"];
            NSBeep();
        }
    } else {
        [_alertSoundName setString:@"Beep"];
        NSBeep();
    }
}

- (IBAction)setAlertWithMessage:(id)sender
{
    _alertWithMessage = [sender state];
    [_autoCloseAlertWindowButton setEnabled:((_alertOnChange && _alertWithMessage) ? YES : NO)];
}

- (IBAction)setAutoCloseAlertWindow:(id)sender
{
    _autoCloseAlertWindow = [sender state];
}

- (IBAction)setAlertWithEmail:(id)sender
{
    _alertWithEmail = [sender state];
    [_emailAddressField setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
    [_emailShortMessagesButton setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
}

- (IBAction)setEmailShortMessages:(id)sender
{
    _emailShortMessages = [sender state];
}

- (void)turnOffAlertWithMessage:(BOOL)state
{
    _alertWithMessage = state;
    // we are setting this option from the alert window, so here we have to update the prefs panel.
    [_alertWithMessageButton setEnabled:_alertOnChange];
    [_alertWithMessageButton setState:(_alertWithMessage ? NSOnState : NSOffState)];
    [_autoCloseAlertWindowButton setEnabled:((_alertOnChange && _alertWithMessage) ? YES : NO)];
    [_emailAddressField setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
    [_emailShortMessagesButton setEnabled:((_alertOnChange && _alertWithEmail) ? YES : NO)];
}

- (IBAction)setCurrentAEItemFromPrefs:(id)sender
{
    _currentAEItem = [[sender selectedItem] tag];
    [self updateCosmicMenu];
}

- (IBAction)changeTimeoutValue:(id)sender
{
    _timeout = [sender intValue];
    [_timeoutTextField setIntValue:_timeout];
}

- (IBAction)setStatusItemStyle:(id)sender
{
    _statusItemStyle = [[sender selectedItem] tag];
    [self updateCosmicMenu];
}

- (IBAction)addOrRemoveToLoginItems:(id)sender
{
    if ([sender state]) {
        if (![self isLoginItem]) [self setLoginItem];
    } else {
        if ([self isLoginItem]) [self removeLoginItem];
    }
}













#pragma mark
#pragma mark SETUP & STATE METHODS
#pragma mark

- (void)loadDownloadingIcons
{
    int i;
    for (i = 0; i < 12; i++) {
        [_statusDLIcons addObject:[NSImage imageNamed:[NSString stringWithFormat:@"SpinnerS%d.png", i]]];
    }
}

- (void)compileSystemSoundsMenu
{
    [_alertSoundsMenu removeAllItems];
    [_alertSoundsMenu addItemWithTitle:@"Beep"];
    
    id fileManager = [NSFileManager defaultManager]; 
    id libraries = [NSArray arrayWithObjects:@"/System/Library/Sounds/",@"Library/Sounds/",@"~/Library/Sounds/",nil]; 
    
    NSEnumerator *libEnum = [libraries objectEnumerator];
    id libObj;
    while (libObj = [libEnum nextObject]) {
        NSArray *pathLists = [fileManager directoryContentsAtPath:libObj];
        NSEnumerator *pathEnum = [pathLists objectEnumerator];
        id soundObj;
        while (soundObj = [pathEnum nextObject]) {
            [_alertSoundsMenu addItemWithTitle:[soundObj stringByDeletingPathExtension]];
        }
    }
    [_alertSoundsMenu selectItemWithTitle:_alertSoundName];
}

- (void)initStatusItem
{
    _statusItem = [[[NSStatusBar systemStatusBar] statusItemWithLength:NSVariableStatusItemLength] retain];
    [_statusItem setHighlightMode: YES];
    [_statusItem setEnabled: YES];
    [_statusItem setMenu: _cosmicMenu];
}

- (void)setLastUpdate
{
    if (_lastUpdatedDate) [_lastUpdatedDate release];
    _lastUpdatedDate = nil;
    _lastUpdatedDate = [[NSCalendarDate calendarDate] retain];
    [_lastUpdatedString setString:[_lastUpdatedDate descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    [self calculateNextUpdate];
}

- (int)nextUpdateInMinutes
{
    int timeForUpdate = 0;
    
    switch(_autoUpdateRate) {
        case 0: /* Five Minutes */
            timeForUpdate = 5; /* DEBUG SET BACK TO 5 */
        break;
        
        case 1: /* Five Minutes */
            timeForUpdate = 10;
        break;
        
        case 2: /* 15 Minutes */
            timeForUpdate = 15;
        break;
        
        case 3: /* 30 Minutes */
            timeForUpdate = 30;
        break;
        
        case 4: /* 1 Hour */
            timeForUpdate = 60;
        break;
        
        case 5: /* 2 Hours */
            timeForUpdate = 120;
        break;
        
        case 6: /* 6 Hours */
            timeForUpdate = 360;
        break;
        
        case 7: /* 12 Hours */
            timeForUpdate = 720;
        break;
        
        case 8: /* 24 hours */
            timeForUpdate = 1440;
        break;
        
        default:
            timeForUpdate = 15;
        break;
    }
    return timeForUpdate;
}

- (void)calculateNextUpdate
{
    if (_nextUpdateDate) [_nextUpdateDate release];
    _nextUpdateDate = nil;
    NSCalendarDate *nextDate = [_lastUpdatedDate dateByAddingYears:0 months:0 days:0 hours:0 minutes:[self nextUpdateInMinutes] seconds:0];
    [_nextUpdateString setString:[nextDate descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    _nextUpdateDate = [nextDate retain];
}

- (IBAction)doNothing:(id)sender
{
    /* NOP */
}












#pragma mark
#pragma mark INTERFACE UPDATE METHODS
#pragma mark

- (void)updateCosmicMenu
{
    
    NSImage *statusImage;
    if (_currentAEItem == 0) {
        statusImage = [NSImage imageNamed:[NSString stringWithFormat:@"Menu%d_%d.png", _statusItemStyle, _currentHighLat]];
        [_statusItem setToolTip:[NSString stringWithFormat:@"Chance for Northern Lights: %@", [self stringForProbabilityLevel:_currentHighLat]]];
    }
    if (_currentAEItem == 1) {
        statusImage = [NSImage imageNamed:[NSString stringWithFormat:@"Menu%d_%d.png", _statusItemStyle, _currentMidLat]];
        [_statusItem setToolTip:[NSString stringWithFormat:@"Chance for Northern Lights: %@", [self stringForProbabilityLevel:_currentMidLat]]];
    }
    if (_currentAEItem == 2) {
        statusImage = [NSImage imageNamed:[NSString stringWithFormat:@"Menu%d_%d.png", _statusItemStyle, _currentLowLat]];
        [_statusItem setToolTip:[NSString stringWithFormat:@"Chance for Northern Lights: %@", [self stringForProbabilityLevel:_currentLowLat]]];
    }
    if (_currentAEItem == 3) {
        statusImage = [NSImage imageNamed:[NSString stringWithFormat:@"Menu%d_%d.png", _statusItemStyle, _currentPCAFlevel]];
        [_statusItem setToolTip:[NSString stringWithFormat:@"Chance for Northern Lights: %@", [self stringForProbabilityLevel:_currentPCAFlevel]]];
    }
    
    [_statusItem setImage:statusImage];
    
    [_latStatHighMenuItem setTitle:[NSString stringWithFormat:@" High Latitudes: %@", [self stringForProbabilityLevel:_currentHighLat]]];
    [_latStatMidMenuItem setTitle:[NSString stringWithFormat:@" Middle Latitudes: %@", [self stringForProbabilityLevel:_currentMidLat]]];
    [_latStatLowMenuItem setTitle:[NSString stringWithFormat:@" Low Latitudes: %@", [self stringForProbabilityLevel:_currentLowLat]]];
    [_pcafMenuItem setTitle:[NSString stringWithFormat:@" PCAF Level: %@", [self stringForProbabilityLevel:_currentPCAFlevel]]];
    [_latStatHighMenuItem setState:((_currentAEItem == 0) ? NSOnState : NSOffState)];
    [_latStatMidMenuItem setState:((_currentAEItem == 1) ? NSOnState : NSOffState)];
    [_latStatLowMenuItem setState:((_currentAEItem == 2) ? NSOnState : NSOffState)];
    [_pcafMenuItem setState:((_currentAEItem == 3) ? NSOnState : NSOffState)];
    
    [_lastUpdatedMenuItem setTitle:[NSString stringWithFormat:@"Last Update: %@", _lastUpdatedString]];
    if (_autoUpdates) {
        [_nextUpdateMenuItem setTitle:[NSString stringWithFormat:@"Next Update: %@", _nextUpdateString]];
    } else {
        [_nextUpdateMenuItem setTitle:@"Next Update: not scheduled"];
    }
    
    if (_alertOnChange) {
        if (_alertOnAllItemChanges) {
            [self checkForAnyProbabilityChange];
        } else {
            if (_currentAEItem == 0) [self checkProbabilityChange:_currentHighLat previous:_previousHighLat];
            if (_currentAEItem == 1) [self checkProbabilityChange:_currentMidLat previous:_previousMidLat];
            if (_currentAEItem == 2) [self checkProbabilityChange:_currentLowLat previous:_previousLowLat];
            if (_currentAEItem == 3) [self checkProbabilityChange:_currentPCAFlevel previous:_previousPCAFlevel];
        }
    }
    
    if (_currentHighLat != 0)   _previousHighLat   = _currentHighLat;
    if (_currentMidLat != 0)    _previousMidLat    = _currentMidLat;
    if (_currentLowLat != 0)    _previousLowLat    = _currentLowLat;
    if (_currentPCAFlevel != 0) _previousPCAFlevel = _currentPCAFlevel;
    
}

- (void)checkProbabilityChange:(int)current previous:(int)previous
{
    if ( current > previous) {
        if (current >= (_alertThreshold + 1)) {
            if (_alertWithSound) {
                if ([_alertSoundName isEqualToString:@"Beep"]) {
                    NSBeep();
                } else {
                    NSSound *snd = [NSSound soundNamed:_alertSoundName];
                    if (snd) [snd play];
                }
            }
            if (_alertWithMessage) {
                [_alerter runSingleAlertWindowForLevel:current description:[self stringForProbabilityLevel:current] autoClose:_autoCloseAlertWindow];
            }
            if (_alertWithEmail) [self sendEmailAlertForLat:_currentAEItem value:current];
        }
    }
}

- (void)checkForAnyProbabilityChange
{
    if ((_currentHighLat > _previousHighLat) || 
        (_currentMidLat > _previousMidLat) || 
        (_currentLowLat > _previousLowLat) ||
        (_currentPCAFlevel > _previousPCAFlevel))
    {
        if ((_currentHighLat >= _alertThreshold) || 
        (_currentMidLat >= _alertThreshold) || 
        (_currentLowLat >= _alertThreshold) ||
        (_currentPCAFlevel >= _alertThreshold))
        {
            if (_alertWithSound) {
                if ([_alertSoundName isEqualToString:@"Beep"]) {
                    NSBeep();
                } else {
                    NSSound *snd = [NSSound soundNamed:_alertSoundName];
                    if (snd) [snd play];
                }
            }
            if (_alertWithMessage) {
                [_alerter runMultiAlertWindowWithHighLat:[self stringForProbabilityLevel:_currentHighLat] 
                                    midLat:[self stringForProbabilityLevel:_currentMidLat]
                                    lowLat:[self stringForProbabilityLevel:_currentLowLat] 
                                    PCAF:[self stringForProbabilityLevel:_currentPCAFlevel]
                                    autoClose:_autoCloseAlertWindow];
            }
            if (_alertWithEmail) [self sendEmailAlertForLat:0 value:0];
        }
    }
}

- (IBAction)setCurrentAEItemFromMenu:(id)sender
{
    _currentAEItem = [sender tag];
    [_currentAEItemMenu selectItemAtIndex:_currentAEItem];
    [self updateCosmicMenu];
}












#pragma mark
#pragma mark PARSING METHODS
#pragma mark

- (void)parseData:(NSData *)data forType:(int)type
{
    if (type == 0) { // STD
        _currentHighLat     = 0;
        _currentMidLat      = 0;
        _currentLowLat      = 0;
        NSString *htmlString = [[NSString alloc] initWithBytes:[data bytes] length:[data length] encoding:NSASCIIStringEncoding];
        if (htmlString && [htmlString length] > 0) {
            _currentHighLat = [self getHighLatitudeProbabilityLevelFromString:htmlString];
            _currentMidLat  = [self getMiddleLatitudeProbabilityLevelFromString:htmlString];
            _currentLowLat  = [self getLowLatitudeProbabilityLevelFromString:htmlString];
        }
        [htmlString release];
    } else { // PCAF
        _currentPCAFlevel   = 0;
        NSString *htmlString = [[NSString alloc] initWithBytes:[data bytes] length:[data length] encoding:NSASCIIStringEncoding];
        if (htmlString && [htmlString length] > 0) {
            _currentPCAFlevel = [self getPCAFProbabilityLevelFromString:htmlString];
        }
        [htmlString release];
    }
    return;
}

- (int)getHighLatitudeProbabilityLevelFromString:(NSString *)htmlString
{    
    if (htmlString && [htmlString length] > 0) {
        NSString *highTag = nil;
        NSScanner *scanner = [NSScanner scannerWithString:htmlString];
        [scanner scanUpToString:@"High Latitudes:" intoString:nil];
        [scanner scanString:@"High Latitudes:" intoString:nil];
        [scanner scanString:@"<img src=\"" intoString:nil];
        [scanner scanUpToString:@"\"" intoString:&highTag];
        if (highTag && [highTag length] > 0) {
            if ([highTag caseInsensitiveCompare:@"Greenlight.gif"] == NSOrderedSame) return 1;
            if ([highTag caseInsensitiveCompare:@"Yellowlight.gif"] == NSOrderedSame) return 2;
            if ([highTag caseInsensitiveCompare:@"Redlight.gif"] == NSOrderedSame) return 3;
            return 0;
        } else {
            return 0;
        }
    } else {
        return 0;
    }
}

- (int)getMiddleLatitudeProbabilityLevelFromString:(NSString *)htmlString
{
    if (htmlString && [htmlString length] > 0) {
        NSString *midTag = nil;
        NSScanner *scanner = [NSScanner scannerWithString:htmlString];
        [scanner scanUpToString:@"Middle Latitudes:" intoString:nil];
        [scanner scanString:@"Middle Latitudes:" intoString:nil];
        [scanner scanString:@"<img src=\"" intoString:nil];
        [scanner scanUpToString:@"\"" intoString:&midTag];
        if (midTag && [midTag length] > 0) {
            if ([midTag caseInsensitiveCompare:@"Greenlight.gif"] == NSOrderedSame) return 1;
            if ([midTag caseInsensitiveCompare:@"Yellowlight.gif"] == NSOrderedSame) return 2;
            if ([midTag caseInsensitiveCompare:@"Redlight.gif"] == NSOrderedSame) return 3;
            return 0;
        } else {
            return 0;
        }
    } else {
        return 0;
    }
}

- (int)getLowLatitudeProbabilityLevelFromString:(NSString *)htmlString
{
    if (htmlString && [htmlString length] > 0) {
        NSString *lowTag = nil;
        NSScanner *scanner = [NSScanner scannerWithString:htmlString];
        [scanner scanUpToString:@"Low Latitudes:" intoString:nil];
        [scanner scanString:@"Low Latitudes:" intoString:nil];
        [scanner scanString:@"<img src=\"" intoString:nil];
        [scanner scanUpToString:@"\"" intoString:&lowTag];
        if (lowTag && [lowTag length] > 0) {
            if ([lowTag caseInsensitiveCompare:@"Greenlight.gif"] == NSOrderedSame) return 1;
            if ([lowTag caseInsensitiveCompare:@"Yellowlight.gif"] == NSOrderedSame) return 2;
            if ([lowTag caseInsensitiveCompare:@"Redlight.gif"] == NSOrderedSame) return 3;
            return 0;
        } else {
            return 0;
        }
    } else {
        return 0;
    }
}

- (int)getPCAFProbabilityLevelFromString:(NSString *)htmlString
{
    if (htmlString && [htmlString length] > 0) {
        NSString *pcafTag = nil;
        NSScanner *scanner = [NSScanner scannerWithString:htmlString];
        [scanner scanUpToString:@"PCAF" intoString:nil];
        [scanner scanString:@"PCAF" intoString:nil];
        [scanner scanUpToString:@"\n" intoString:&pcafTag];
        if (pcafTag && [pcafTag length] > 0) {
            if ([pcafTag caseInsensitiveCompare:@"green"] == NSOrderedSame) return 1;
            if ([pcafTag caseInsensitiveCompare:@"yellow"] == NSOrderedSame) return 2;
            if ([pcafTag caseInsensitiveCompare:@"red"] == NSOrderedSame) return 3;
            if ([pcafTag caseInsensitiveCompare:@"in progress"] == NSOrderedSame) return 4;
            return 0;
        } else {
            return 0;
        }
    } else {
        return 0;
    }
}

- (NSString *)stringForProbabilityLevel:(int)level
{
    if (level == 0) return @"Unknown";
    if (level == 1) return @"Low";
    if (level == 2) return @"Medium";
    if (level == 3) return @"High";
    if (level == 4) return @"Active"; // For PCAF
    return @"Unknown";
}

- (NSString *)stringForAEItemIndex:(int)index
{
    if (index == 0) return @"High Latitudes";
    if (index == 1) return @"Middle Latitudes";
    if (index == 2) return @"Low Latitudes";
    if (index == 3) return @"PCAF";
    return @"Unknown";
}









- (void)setupCacheFolder
{
    NSString *cacheFolderName = [NSString stringWithString:@"Cosmic Debris"];
    _cacheFolderPath = [[self pathFromUserLibraryPath:cacheFolderName] retain];
    if (! [self checkCreateDataFolder:cacheFolderName] ) {
        NSLog(@"Can't create Cosmic Debris cache folder");
        [_cacheFolderPath release];
        _cacheFolderPath = nil;
        return;
    }
}






#pragma mark
#pragma mark AURORA OVAL IMAGE METHODS
#pragma mark

- (void)loadCachedOvalImage
{
    NSString *filePath = [_cacheFolderPath stringByAppendingPathComponent: _ovalFileName];
    if ([[NSFileManager defaultManager] fileExistsAtPath: filePath]) {
        NSImage *image = [[[NSImage alloc] initWithContentsOfFile: filePath] autorelease];
        [self setOvalImageView:image];
    } else {
        [self clearOvalImage];
    }
}

- (void)updateOvalImageFromData:(NSData *)data
{
    if (data == nil || [data length] == 0) {
        [self clearOvalImage];
        return;
    }
    
    [_lastOvalImageUpdate setString:[[NSCalendarDate calendarDate] descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    NSImage *image = [[[NSImage alloc] initWithData: data] autorelease];
    [self setOvalImageView:image];
    if (_cacheFolderPath && [_cacheFolderPath length] != 0) {
        if (image) {
            NSString *imageFilePath = [_cacheFolderPath stringByAppendingPathComponent: _ovalFileName];
            [data writeToFile:imageFilePath atomically:YES];
        }
    }
    return;
}


- (void)clearOvalImage
{
    [_ovalWindow setImage:nil updated:_lastOvalImageUpdate];
    return;
}

- (void)setOvalImageView:(NSImage *)image
{
    [_ovalWindow setImage:image updated:_lastOvalImageUpdate];
    return;
}







#pragma mark
#pragma mark KP IMAGE METHODS
#pragma mark

- (void)loadCachedKPImage
{
    NSString *filePath = [_cacheFolderPath stringByAppendingPathComponent: _KPFileName];
    if ([[NSFileManager defaultManager] fileExistsAtPath: filePath]) {
        NSImage *image = [[[NSImage alloc] initWithContentsOfFile: filePath] autorelease];
        [self setKPImageView:image];
    } else {
        [self clearKPImage];
    }
}

- (void)updateKPImageFromData:(NSData *)data
{
    if (data == nil || [data length] == 0) {
        [self clearKPImage];
        return;
    }
    
    [_lastKPImageUpdate setString:[[NSCalendarDate calendarDate] descriptionWithCalendarFormat:@"%m/%d %I:%M%p"]];
    NSImage *image = [[[NSImage alloc] initWithData: data] autorelease];
    [self setKPImageView:image];
    if (_cacheFolderPath && [_cacheFolderPath length] != 0) {
        if (image) {
            NSString *imageFilePath = [_cacheFolderPath stringByAppendingPathComponent: _KPFileName];
            [data writeToFile:imageFilePath atomically:YES];
        }
    }
    return;
}

- (void)clearKPImage
{
    [_KPWindow setImage:nil updated:_lastKPImageUpdate];
    return;
}

- (void)setKPImageView:(NSImage *)image
{
    [_KPWindow setImage:image updated:_lastKPImageUpdate];
    return;
}








#pragma mark
#pragma mark DOWNLOADING METHODS
#pragma mark

- (IBAction)updateOrCancel:(id)sender
{
    if ([self isDownloading]) {
        [self cancel:nil];
    } else {
        [self update:nil];
    }
}

- (IBAction)update:(id)sender
{
    if ([self isDownloading]) [self cancel:nil];
    [_statusItem setToolTip:[NSString stringWithFormat:@"%@ %@", _appNameString, _versionString]];
    [_cancelOrUpdateMenuItem setTitle:@"Cancel Downloading"];
    [self startAnimationTimer];
    [_STDDownloader downloadDataWithTimeoutInterval:_timeout];
    [_PCAFDownloader downloadDataWithTimeoutInterval:_timeout];
    if (_downloadOvalImage) [_ovalDownloader downloadDataWithTimeoutInterval:_timeout];
    if (_downloadKPImage)   [_KPDownloader downloadDataWithTimeoutInterval:_timeout];
}

- (IBAction)cancel:(id)sender
{
    if ([_STDDownloader isDownloading])  [_STDDownloader cancelDownloading];
    if ([_PCAFDownloader isDownloading]) [_PCAFDownloader cancelDownloading];
    if ([_ovalDownloader isDownloading]) [_ovalDownloader cancelDownloading];
    if ([_KPDownloader isDownloading])   [_KPDownloader cancelDownloading];
}

- (BOOL)checkAutoUpdate
{
    if (!_autoUpdates) return NO;
    
    NSCalendarDate *now = [NSCalendarDate calendarDate];
    
    if (_nextUpdateDate) {
        if ([_nextUpdateDate isEqualToDate:now] || [now isEqualToDate:[now laterDate:_nextUpdateDate]]) {
            return YES;
        } else {
            return NO;
        }
    } else {
        return YES;
    }
}

- (void)checkDownloadAtStartup
{
    if (!_downloadAtStartup) return;
    if ([self checkAutoUpdate]) return; // gonna dl, skip this
    if ([self isDownloading]) return;
    [self update:nil];
}

- (BOOL)isDownloading
{
    if ([_STDDownloader isDownloading]) return YES;
    if ([_PCAFDownloader isDownloading]) return YES;
    if ([_ovalDownloader isDownloading] && _downloadOvalImage) return YES;
    if ([_KPDownloader isDownloading]   && _downloadKPImage) return YES;
    return NO;
}









#pragma mark
#pragma mark TIMER METHODS
#pragma mark

- (void)startUpdateTimer
{
    [self stopUpdateTimer];
    
    NSTimer *timer = [NSTimer scheduledTimerWithTimeInterval:5.0
                            target:self 
                            selector:@selector(updateTimer:) 
                            userInfo:nil 
                            repeats:YES];
    if (timer) _updateTimer = [timer retain];
    [self updateTimer:nil]; // go ahead and check right away
}

- (void)stopUpdateTimer
{
    if (_updateTimer) [_updateTimer invalidate];
    [_updateTimer release];
    _updateTimer = nil;
}

- (void)updateTimer:(NSTimer *)timer
{
    if ([self checkAutoUpdate]) [self update:nil];
}

- (void)startAnimationTimer
{
    _statusItemIconIndex = 0;
    [self stopAnimationTimer];
    
    NSTimer *timer = [NSTimer scheduledTimerWithTimeInterval:(float)STATUS_ICON_ANIMATION_SPEED_SECONDS
                            target:self 
                            selector:@selector(animationTimer:) 
                            userInfo:nil 
                            repeats:YES];
    if (timer) _animationTimer = [timer retain];
}

- (void)stopAnimationTimer
{
    if (_animationTimer) [_animationTimer invalidate];
    [_animationTimer release];
    _animationTimer = nil;
}

- (void)animationTimer:(NSTimer *)timer
{
    if (_statusItemIconIndex >= [_statusDLIcons count]) return;
    
    [_statusItem setImage:[_statusDLIcons objectAtIndex:_statusItemIconIndex]];
    _statusItemIconIndex++;
    if (_statusItemIconIndex >= [_statusDLIcons count]) {
        _statusItemIconIndex = 0;
    }
}


















#pragma mark
#pragma mark DOWNLOADER DELEGATE METHODS
#pragma mark

- (void)downloaderDidBegin:(Downloader *)downloader
{
    /* NOP */
}

- (void)downloaderDidConnect:(Downloader *)downloader expectedLength:(int)expectedLength
{
    /* NOP */
}

- (void)downloaderDidFailWithError:(Downloader *)downloader error:(NSString *)error
{
    NSLog(error);
    if (downloader == _STDDownloader) {
        _currentHighLat     = 0;
        _currentMidLat      = 0;
        _currentLowLat      = 0;
    }
    if (downloader == _PCAFDownloader) _currentPCAFlevel = 0;
}

- (void)downloaderDidReceiveData:(Downloader *)downloader bytesReceived:(int)bytesReceived expectedLength:(int)expectedLength
{
    if (expectedLength > 0) {
        double percentComplete = ((double)bytesReceived / (double)expectedLength) * 100;
    }
}

- (void)downloaderDidFinishWithData:(Downloader *)downloader data:(NSData *)data
{
    if (downloader == _STDDownloader)  [self parseData:data forType:0];
    if (downloader == _PCAFDownloader) [self parseData:data forType:1];
    if (downloader == _ovalDownloader) [self updateOvalImageFromData: data];
    if (downloader == _KPDownloader)   [self updateKPImageFromData: data];
}

- (void)downloaderDidReset:(Downloader *)downloader
{
    // Finished, canceled, error, whatever, this method is *always* called when downloader stops.
    // it's the perfect place to update the interface and state.
    if (![self isDownloading]) {
        [self setLastUpdate];
        [self stopAnimationTimer];
        [self updateCosmicMenu];
        [_cancelOrUpdateMenuItem setTitle:@"Refresh Data..."];
    }
}














#pragma mark
#pragma mark FOO METHODS
#pragma mark

- (IBAction)openAboutBoxen:(id)sender
{
    [self setLastApplicationPath];
    [NSApp activateIgnoringOtherApps:YES];
    [_aboutBoxen center];
    [_aboutBoxen makeKeyAndOrderFront:self];
}

- (IBAction)linkClicked:(id)sender
{
    NSURL *url = [NSURL URLWithString: [sender stringValue]];
    if (url == nil) return;
    [[NSWorkspace sharedWorkspace] openURL:url];
    [_aboutBoxen close];
}

- (IBAction)mailLinkClicked:(id)sender
{
    [[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:@"mailto:john@jschilling.net?subject=Cosmic%20Debris"]];
    [_aboutBoxen close];
}

- (IBAction)donateLinkClicked:(id)sender
{
    [[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:@"https://www.paypal.com/xclick/business=john%40jschilling.net&no_note=1&tax=0&currency_code=USD"]];
    [_aboutBoxen close];
}

- (IBAction)openMoreInfoLink:(id)sender
{
    NSString *str;
    switch ([sender tag]) {
        case 0:
            str = @"http://www.spacew.com/";
        break;
        
        case 1:
            str = @"http://www.spacew.com/www/aurora.html";
        break;
        
        case 2:
            str = @"http://www.sec.noaa.gov/forecast.html";
        break;
        
        case 3:
            str = @"http://www.sec.noaa.gov/pmap/";
        break;
    }
    NSURL *url = [NSURL URLWithString: str];
    if (url == nil) return;
    [[NSWorkspace sharedWorkspace] openURL:url];
}

- (void)sendEmailAlertForLat:(int)latNum value:(int)value
{
    if (!_emailAddress || [_emailAddress length] == 0) return;
    if (!_alertWithEmail) return;
    
    NSAttributedString *theMessage;
    
    if (_alertOnAllItemChanges) {
        theMessage = [[[NSAttributedString alloc] initWithString:[self createEmailAllMessageFromFile]] autorelease];
    } else {
        theMessage = [[[NSAttributedString alloc] initWithString:[self createEmailMessageFromFileForLat:latNum value:value]] autorelease];
    }
    if (!theMessage || [theMessage length] == 0) return;
    
    NSMutableDictionary *headersDict = [[NSMutableDictionary alloc] init]; 
    [headersDict setObject:_emailAddress forKey:@"To"];
    if (_emailShortMessages) {
        [headersDict setObject:@"CD Aurora Alert" forKey:@"Subject"];
    } else {
        [headersDict setObject:@"Cosmic Debris Aurora Alert" forKey:@"Subject"];
    }
    
    id emailer = [[[EmailThread alloc] init] autorelease];
    [NSThread detachNewThreadSelector:@selector(sendEmail:) toTarget:emailer 
        withObject:[NSArray arrayWithObjects:headersDict , theMessage, nil]];
    [headersDict release];
}

- (NSString *)createEmailAllMessageFromFile
{
    NSString *emailPath;
    if (_emailShortMessages) {
        emailPath = [NSString stringWithFormat:@"%@/emailAllShort.txt", [[NSBundle mainBundle] resourcePath]];
    } else {
        emailPath = [NSString stringWithFormat:@"%@/emailAllLong.txt", [[NSBundle mainBundle] resourcePath]];
    }
    NSMutableString *messageBody = [NSMutableString stringWithContentsOfFile:emailPath];
    
    if (messageBody && [messageBody length] > 0) {
        NSString *highLat = [self stringForProbabilityLevel:_currentHighLat];
        NSString *midLat = [self stringForProbabilityLevel:_currentMidLat];
        NSString *lowLat = [self stringForProbabilityLevel:_currentLowLat];
        NSString *PCAF = [self stringForProbabilityLevel:_currentPCAFlevel];
        [messageBody replaceOccurrencesOfString:@"<timestamp>" withString:_lastUpdatedString options:nil range:NSMakeRange(0, [messageBody length])];
        [messageBody replaceOccurrencesOfString:@"<high>" withString:highLat options:nil range:NSMakeRange(0, [messageBody length])];
        [messageBody replaceOccurrencesOfString:@"<middle>" withString:midLat options:nil range:NSMakeRange(0, [messageBody length])];
        [messageBody replaceOccurrencesOfString:@"<low>" withString:lowLat options:nil range:NSMakeRange(0, [messageBody length])];
        [messageBody replaceOccurrencesOfString:@"<PCAF>" withString:PCAF options:nil range:NSMakeRange(0, [messageBody length])];
    }
    return messageBody;
}

- (NSString *)createEmailMessageFromFileForLat:(int)latNum value:(int)value
{
    NSString *emailPath;
    if (_emailShortMessages) {
        emailPath = [NSString stringWithFormat:@"%@/emailOneShort.txt", [[NSBundle mainBundle] resourcePath]];
    } else {
        emailPath = [NSString stringWithFormat:@"%@/emailOneLong.txt", [[NSBundle mainBundle] resourcePath]];
    }
    NSMutableString *messageBody = [NSMutableString stringWithContentsOfFile:emailPath];
    
    if (messageBody && [messageBody length] > 0) {
        NSString *latName = [self stringForAEItemIndex:latNum];
        NSString *vString = [self stringForProbabilityLevel:value];
        [messageBody replaceOccurrencesOfString:@"<timestamp>" withString:_lastUpdatedString options:nil range:NSMakeRange(0, [messageBody length])];
        [messageBody replaceOccurrencesOfString:@"<latname>" withString:latName options:nil range:NSMakeRange(0, [messageBody length])];
        [messageBody replaceOccurrencesOfString:@"<value>" withString:vString options:nil range:NSMakeRange(0, [messageBody length])];
    }
    return messageBody;
}

- (BOOL)isLoginItem
{
    id obj;

    NSString *ourAppsPath = [[NSBundle mainBundle] bundlePath];
    NSDictionary *loginItemDict = 
        [NSDictionary dictionaryWithContentsOfFile: [NSString stringWithFormat: @"%@/Library/Preferences/loginwindow.plist", NSHomeDirectory()]];
    NSEnumerator *loginItemEnumerator = 
        [[loginItemDict objectForKey:
            @"AutoLaunchedApplicationDictionary"] objectEnumerator];

    while (( obj = [loginItemEnumerator nextObject] ) != nil )
    {
        if ( [[obj objectForKey:@"Path"] isEqualTo:ourAppsPath] )
            return( YES );
    }
    return( NO );
}

- (void)setLoginItem
{
    NSMutableArray *loginItems;
    
    loginItems = (NSMutableArray *)CFPreferencesCopyValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" ,
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    loginItems = [[loginItems autorelease] mutableCopy]; 
    
    NSMutableDictionary *myDict = [[[NSMutableDictionary alloc] init] autorelease];
    [myDict setObject:[NSNumber numberWithBool:NO] forKey:@"Hide"];
    [myDict setObject:[[NSBundle mainBundle] bundlePath] forKey:@"Path"];
    
    [loginItems removeObject:myDict]; //make sure it's not already in there 
    [loginItems addObject:myDict]; 
    
    CFPreferencesSetValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" , 
        loginItems , 
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    CFPreferencesSynchronize(
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    
    [loginItems release];
}

- (void)removeLoginItem
{
    NSMutableArray *loginItems;
    
    loginItems = (NSMutableArray *)CFPreferencesCopyValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" ,
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    loginItems = [[loginItems autorelease] mutableCopy];
    
    NSMutableDictionary *myDict = [[[NSMutableDictionary alloc] init] autorelease];
    [myDict setObject:[NSNumber numberWithBool:NO] forKey:@"Hide"];
    [myDict setObject:[[NSBundle mainBundle] bundlePath] forKey:@"Path"];
    
    [loginItems removeObject:myDict];
    
    CFPreferencesSetValue(
        (CFStringRef)@"AutoLaunchedApplicationDictionary" , 
        loginItems , 
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    CFPreferencesSynchronize(
        (CFStringRef)@"loginwindow" , 
        kCFPreferencesCurrentUser , 
        kCFPreferencesAnyHost ); 
    
    [loginItems release];
}






#pragma mark 
#pragma mark DATA FOLDER METHODS
#pragma mark 

- (BOOL)checkCreateDataFolder:(NSString *)path
{
    NSString *folderPath = [self pathFromUserLibraryPath:path];
    NSFileManager *manager = [NSFileManager defaultManager];
    
    if ( ![self ifFolderExists:folderPath] )
    {
        return [manager createDirectoryAtPath:folderPath attributes:nil];
    } else {
        return YES;
    }
}

- (BOOL)ifFolderExists:(NSString *)path
{
    BOOL isDir = YES;
    NSFileManager *manager = [NSFileManager defaultManager];
    return [manager fileExistsAtPath:path isDirectory:&isDir];
}

- (NSString *)pathFromUserLibraryPath:(NSString *)inSubPath
{
    //
    // Thanks to Karelia Software for this method. 
    // http://cocoa.karelia.com/
    //
    
    NSArray *domains = NSSearchPathForDirectoriesInDomains(NSLibraryDirectory,NSUserDomainMask,YES);
    NSString *baseDir= [domains objectAtIndex:0];
    NSString *result = [baseDir stringByAppendingPathComponent:inSubPath];
    return result;
}



- (void)setLastApplicationPath
{
    NSDictionary *appDict = [[NSWorkspace sharedWorkspace] activeApplication];
    if (appDict == nil) {
        _lastAppPath = nil;
        return;
    }
    NSString *myPath = [[NSBundle mainBundle] bundlePath];
    NSString *appPath = [appDict objectForKey:@"NSApplicationPath"];
    if (![myPath isEqualToString:appPath]) {
        [_lastAppPath setString: appPath];
    }
}

- (void)returnFocusToLastApplication
{
    if (_lastAppPath == nil || ([_lastAppPath length] < 1)) return;
    NSArray *activeApps = [[NSWorkspace sharedWorkspace] launchedApplications];
    
    NSEnumerator *appsEnum = [activeApps objectEnumerator];
    id anApp;
    while (anApp = [appsEnum nextObject]) {
        NSString *appPath = [anApp objectForKey:@"NSApplicationPath"];
        NSString *appName = [anApp objectForKey:@"NSApplicationName"];
        if ([appPath isEqualToString:_lastAppPath] && ![appName isEqualToString:@"Finder"]) {
            [[NSWorkspace sharedWorkspace] launchApplication: _lastAppPath];
        }
    }
}






#pragma mark 
#pragma mark WINDOW DELEGATE METHODS
#pragma mark

- (BOOL)windowShouldClose:(id)sender
{
    [self returnFocusToLastApplication];
    
    if (sender == _prefsWindow) {
        NSString *email = [_emailAddressField stringValue];
        NSURL *url = [NSURL URLWithString:email];
        if (url) {
            [_emailAddress setString:email];
        }
    } else if (sender == _ovalWindow) {
        _ovalWindowOpen = NO;
    } else if (sender == _KPWindow) {
        _KPWindowOpen = NO;
    }
    
    return YES;
}

- (void)windowDidBecomeKey:(NSNotification *)aNotification
{
    if ([aNotification object] == _ovalWindow) {
        _ovalWindowOpen = YES;
    } else if ([aNotification object] == _KPWindow) {
        _KPWindowOpen = YES;
    }
}

- (void)windowDidMiniaturize:(NSNotification *)aNotification
{
    if ([aNotification object] == _ovalWindow) {
         _ovalWindowDocked = YES;
    } else if ([aNotification object] == _KPWindow) {
        _KPWindowDocked = YES;
    }
}

- (void)windowDidDeminiaturize:(NSNotification *)aNotification
{
    [[aNotification object] update];
    
    if ([aNotification object] == _ovalWindow) {
        _ovalWindowDocked = NO;
    } else if ([aNotification object] == _KPWindow) {
        _KPWindowDocked = NO;
    }
}

@end
